/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.util;

import java.io.IOException;
import java.io.OutputStream;

/**
 * Provides an output stream wrapper that detects line feed sequences and
 * replaces them by new ones.
 *
 */
public class LineOutputStream extends OutputStream {

    public static final byte[] LS_BINARY = null;

    public static final byte[] LS_UNIX = new byte[]{0x0a};

    public static final byte[] LS_WINDOWS = new byte[]{0x0d, 0x0a};

    public static final byte[] LS_NATIVE = System.getProperty("line.separator").getBytes();

    private byte[] buffer = new byte[8192];

    private byte[] lineFeed = LS_NATIVE;

    private int pos = 0;

    private static final char STATE_INIT = ' ';
    private static final char STATE_CR = 'c';
    private static final char STATE_LF = 'l';
    private static final char STATE_CRLF = 'f';

    private char state = STATE_INIT;

    private final OutputStream out;

    public LineOutputStream(OutputStream out, byte[] ls) {
        this.out = out;
        if (ls != null) {
            this.lineFeed = ls;
        }
    }

    public void write(int b) throws IOException {
        if (b == 0x0a) {
            switch (state) {
                case STATE_INIT:
                    state = STATE_LF;
                    break;
                case STATE_CR:
                    state = STATE_CRLF;
                    break;
                case STATE_LF:
                    flush(true);
                    state = STATE_LF;
                    break;
                case STATE_CRLF:
                    flush(true);
                    state = STATE_LF;
            }
        } else if (b == 0x0d) {
            switch (state) {
                case STATE_INIT:
                    state = STATE_CR;
                    break;
                case STATE_LF:
                    state = STATE_CRLF;
                    break;
                case STATE_CR:
                    flush(true);
                    state = STATE_CR;
                    break;
                case STATE_CRLF:
                    flush(true);
                    state = STATE_CR;
            }
        } else {
            if (state != STATE_INIT) {
                flush(true);
                state = STATE_INIT;
            }
            if (pos == buffer.length) {
                flush();
            }
            buffer[pos++] = (byte) (b & 0xff);
        }
    }

    public void flush(boolean addLF) throws IOException {
        flush();
        if (addLF) {
            out.write(lineFeed);
        }
        out.flush();
    }

    public void flush() throws IOException {
        out.write(buffer, 0, pos);
        pos = 0;
        out.flush();
    }

    public void close() throws IOException {
        // check for pending lfs
        flush(state != STATE_INIT);
        out.close();
    }

}