/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.text;

import java.text.FieldPosition;
import java.text.Format;
import java.text.ParsePosition;
import java.util.Properties;

/**
 * The <code>AutoFormatter</code> class implements the automatic conversion
 * of line endings to <code>&lt;br></code> HTML lists. This is the formatting
 * subset of the UBM feature of the former Communiqu 2 system.
 * <p>
 * This implementation only supports automatically converting bulleted and
 * numbered lists as well as line breaking. Therefore these formatting
 * options are not implemented here : {@link #FORMAT_NOISODEC},
 * {@link #FORMAT_ISOENC}, {@link #FORMAT_AUTOLINK}, and {@link #FORMAT_URLENC}.
 * For replacements, use the {@link CodeISO} and/or <code>java.net.URLEncoder</code>
 * classes.
 */
public class AutoFormatter extends Format {

    //---------- format() modifier constants -----------------------------------

    /**
     * ISO encodes the &lt;, &quot; and &amp; characters.
     * @deprecated not implemented here, use {@link CodeISO} instead.
     */
    public static final int FORMAT_NOISODEC = 0x0100;

    /**
     * ISO encode characters according to HTTP standard.
     * @deprecated not implemented here, use {@link CodeISO} instead.
     */
    public static final int FORMAT_ISOENC = 0x0200;

    /** Automatically create ordered and unordered lists. */
    public static final int FORMAT_AUTOLISTS = 0x0800;

    /**
     * Autmatic link formatting (?).
     * @deprecated this is not implemented and has no effects
     */
    public static final int FORMAT_AUTOLINK = 0x1000;

    /** Automatically convert line breaks to &lt;br> tags */
    public static final int FORMAT_BR = 0x2000;

    /**
     * Encodes characters illegal in URLs.
     * @deprecated not implemented here, use {@link CodeISO} or
     *      <code>java.net.URLEncoder#encode(String)</code> instead.
     */
    public static final int FORMAT_URLENC = 0x4000;

    /** Combination of line break handling and automatic lists. */
    public static final int FORMAT_AUTOBR = FORMAT_BR | FORMAT_AUTOLISTS;

    //---------- default values for AUTOBR handling ----------------------------

    /** The default line beaking tag */
    private static final String DEFAULT_BR = "<br>";

    /** The default opening tag for ordered lists */
    private static final String DEFAULT_TAG_OL_OPEN = "<ol start=\"%s\">";

    /** The default closing tag for ordered lists */
    private static final String DEFAULT_TAG_OL_CLOSE = "</ol>";

    /** The default opening tag for items in ordered lists */
    private static final String DEFAULT_TAG_OL_ITEM_OPEN = "<li>";

    /** The default closing tag for items in ordered lists */
    private static final String DEFAULT_TAG_OL_ITEM_CLOSE = "</li>";

    /** The default opening tag for unordered lists */
    private static final String DEFAULT_TAG_UL_OPEN = "<ul>";

    /** The default closing tag for unordered lists */
    private static final String DEFAULT_TAG_UL_CLOSE = "</ul>";

    /** The default opening tag for items in unordered lists */
    private static final String DEFAULT_TAG_UL_ITEM_OPEN = "<li>";

    /** The default closing tag for items in unordered lists */
    private static final String DEFAULT_TAG_UL_ITEM_CLOSE = "</li>";

    /** The normal text formatting state - no list in action */
    private static final int STATE_NORMAL = 1;

    /** State where a <em>&lt;ol&gt;</em> tag is currently open */
    private static final int STATE_OL = 3;

    /** State where a <em>&lt;ul&gt;</em> tag is currently open */
    private static final int STATE_UL = 4;

    //---------- from configuration --------------------------------------------

    /** The tag used for line breaks, default {@link #DEFAULT_BR} */
    private final String tagBr;

    /** The tag used to start ordered lists, default {@link #DEFAULT_TAG_OL_OPEN} */
    private final String tagOlOpen;

    /** The tag used to end ordered lists, default {@link #DEFAULT_TAG_OL_CLOSE} */
    private final String tagOlClose;

    /** The tag used to start items in ordered lists, default {@link #DEFAULT_TAG_OL_ITEM_OPEN} */
    private final String tagOlItemOpen;

    /** The tag used to end items in ordered lists, default {@link #DEFAULT_TAG_OL_ITEM_CLOSE} */
    private final String tagOlItemClose;

    /** The tag used to start unordered lists, default {@link #DEFAULT_TAG_UL_OPEN} */
    private final String tagUlOpen;

    /** The tag used to end ordered lists, default {@link #DEFAULT_TAG_UL_OPEN} */
    private final String tagUlClose;

    /** The tag used to start items in unordered lists, default {@link #DEFAULT_TAG_UL_ITEM_OPEN} */
    private final String tagUlItemStart;

    /** The tag used to end items in unordered lists, default {@link #DEFAULT_TAG_UL_ITEM_CLOSE} */
    private final String tagUlItemClose;

    /**
     * The default modifiers used by the
     * {@link #format(Object, StringBuffer, FieldPosition )} method.
     */
    private final int DEFAULT_MODS = FORMAT_AUTOBR;

    /**
     * The system default formatter. Use this instance when formatting with
     * no special configuration is needed.
     */
    public static final AutoFormatter DEFAULT_FORMATTER = new AutoFormatter(null);

    /**
     * Creates a new <code>AutoFormatter</code> object with the default
     * configuration.
     *
     * @deprecated as of echidna. To get an instance with default settings use
     *      the {@link #DEFAULT_FORMATTER}.
     */
    public AutoFormatter() {
	this(null);
    }

    /**
     * Creates a new <code>AutoFormatter</code> object with the given
     * configuration.
     * <p>
     * The configuration contained in the <code>config</code> parameter is
     * assumed to have the following structure, where <code>config.getName()</code>
     * would return <em>auto</em> or <code>config.getChild("auto")</code> is
     * used as the configuration (this is the same structure as was used in
     * Communiqu 2 to configure automatic formatting.
     * <pre>
         <auto>
            <br begin="<br/>" />
            <ul begin="<ul>" end="</ul>" >
                <item begin="<li>" end="</li>" />
            </ul>
            <ol begin="<ol start=\"%s\">" end="</ol>" >
                <item begin="<li>" itemend="</li>" />
            </ul>
         </auto>
     * </pre>
     * <p>
     * All values are replaced as-is except for the <em>ol.start</em>
     * attribute which gets the string <em>%s</em> replaced with the number
     * which is used in the input to defined the ordered list. If the <em>%s</em>
     * string is missing that number is of course not inserted.
     *
     * @param config The configuration for the automatic formatting replacement.
     * 		If this is <code>null</code>, the defualt configuration is
     * 		used.
     */
    public AutoFormatter(Properties config) {
	super();

        // assert config is not null
        if (config ==  null) {
            config = new Properties();
        }

        // configure from received or empty config
        tagBr = config.getProperty("/br.begin", DEFAULT_BR);
        tagOlOpen = config.getProperty("/ol.begin", DEFAULT_TAG_OL_OPEN);
        tagOlClose = config.getProperty("/ol.end", DEFAULT_TAG_OL_CLOSE);
        tagOlItemOpen = config.getProperty("/ol/item.begin", DEFAULT_TAG_OL_ITEM_OPEN);
        tagOlItemClose = config.getProperty("/ol/item.end", DEFAULT_TAG_OL_ITEM_CLOSE);
        tagUlOpen = config.getProperty("/ul.begin", DEFAULT_TAG_UL_OPEN);
        tagUlClose = config.getProperty("/ul.end", DEFAULT_TAG_UL_CLOSE);
        tagUlItemStart = config.getProperty("/ul/item.begin", DEFAULT_TAG_UL_ITEM_OPEN);
        tagUlItemClose = config.getProperty("/ul/item.end", DEFAULT_TAG_UL_ITEM_CLOSE);
    }

    //---------- Format abstract methods ---------------------------------------

    /**
     * Formats the object according to the standard modifiers, which are
     * automatic line breaks and list formatting. This implementation only
     * supports strings and completely ignores the pos parameter.
     *
     * @param obj The object to format, which must be a String or a
     * 		<code>ClassCastException</code> will be thrown.
     * @param toAppendTo Where to append the formatted data. If <code>null</code>,
     * 		a new string buffer is allocated.
     * @param pos Formatting position information. Not used or obeyed.
     *
     * @return a <code>StringBuffer</code> containing the formatted string. This
     * 		is either the same as <code>toAppendTo</code> or a newly
     * 		allocated <code>StringBuffer</code> if the parameter is
     * 		<code>null</code>.
     *
     * @throws ClassCastException if the object is not a <code>String</code>.
     */
    public StringBuffer format(Object obj, StringBuffer toAppendTo,
            FieldPosition pos) {
	return format((String)obj, toAppendTo, DEFAULT_MODS);
    }

    /**
     * The <code>AutoFormatter</code> class does not support parsing, so an
     * <code>UnsupportedOperationException</code> is thrown when trying to
     * parse.
     *
     * @param source The source string to parse. Ignored.
     * @param status The position to define parsing. Ignored.
     *
     * @throws UnsupportedOperationException as it is not yet implemented.
     */
    public Object parseObject(String source, ParsePosition status) {
	throw new UnsupportedOperationException("parse not implemented yet");
    }

    //--------- AutoFormatter specialities -------------------------------------

    /**
     * Formats the string according to the modifier set. The modifier set may be
     * any combination of the FORMAT constants defined.
     *
     * @param str The string to be formatted.
     * @param mods The modifier set controlling the format operation.
     *
     * @return the formatted string.
     */
    public String format(String str, int mods) {
	return format(str, new StringBuffer(), mods).toString();
    }

    /**
     * Formats the string according to the modifier set. The modifier set may be
     * any combination of the FORMAT constants defined.
     * <p>
     * <b>Implementation Note</b>: Only line break processing and automatic
     * 		lists are currently supported.
     *
     * @param str The string to be formatted.
     * @param toAppendTo The string buffer to append the result to. If this
     * 		<code>null</code> a new string buffer is created and returned,
     * 		else the result of the method will be this string buffer.
     * @param mods The modifier set controlling the format operation.
     *
     * @return the string buffer to which the formatted result has been
     * 		appended. This is either the input string buffer or a newly
     * 		created string buffer, if null has been supplied.
     */
    public StringBuffer format(String str, StringBuffer toAppendTo, int mods) {

	int beginCopy = 0;
	int state = STATE_NORMAL;
	boolean lineStart = true;
	char[] buffer = str.toCharArray();
	int bufferLen = buffer.length;

	// assert the string buffer
	if (toAppendTo == null) toAppendTo = new StringBuffer();

	for (int i=0; i<bufferLen; /* i incr. within loop */ ) {
	    char c = buffer[i];

	    // Beginning of line - is it autolist ?
	    if (lineStart && (mods&FORMAT_AUTOLISTS) != 0) {

		// of course this terminates the line start
		lineStart = false;

		int ignore = 0;
		if (c == '-' && i+1<bufferLen && buffer[i+1] == ' ') {
		    // this is ul item

		    // end possibly open OL list
		    if (state == STATE_OL) {
			toAppendTo.append(tagOlItemClose);
			toAppendTo.append(tagOlClose);
		    }

		    // if UL list is not yet open, start, else end previous item
		    if (state != STATE_UL) {
			toAppendTo.append(tagUlOpen);
			state = STATE_UL;
		    } else {
			toAppendTo.append(tagUlItemClose);
		    }

		    // start this item
		    toAppendTo.append(tagUlItemStart);

		    // we used two characters : '- '
		    ignore = 2;

		} else {
		    // check for number at the beginning of the line
		    StringBuffer start = new StringBuffer();
		    int j;
		    for (j=i; j+2<bufferLen && c>='0' && c<='9'; c=buffer[++j]) {
			start.append(c);
		    }

		    if (j>i && j+1<bufferLen && buffer[j]=='.' && buffer[j+1]==' ') {
			// this is ol item

			// end possibly open UL list
			if (state == STATE_UL) {
			    toAppendTo.append(tagUlItemClose);
			    toAppendTo.append(tagUlClose);
			}

			// if OL list is not yet open, start, else end previous item
			if (state != STATE_OL) {
			    toAppendTo.append(Text.sprintf(tagOlOpen, start.toString()));
			    state = STATE_OL;
			} else {
			    toAppendTo.append(tagOlItemClose);
			}

			// start this item
			toAppendTo.append(Text.sprintf(tagOlItemOpen, Integer.valueOf(start.toString())));

			// we used the number plus '. '
			ignore = j-i+2;

		    } else {
			// not a list item (anymore) - close open lists

			if (state == STATE_UL) {
			    toAppendTo.append(tagUlItemClose);
			    toAppendTo.append(tagUlClose);
			    state = STATE_NORMAL;
			} else if (state == STATE_OL) {
			    toAppendTo.append(tagOlItemClose);
			    toAppendTo.append(tagOlClose);
			    state = STATE_NORMAL;
			}
		    }
		}

		// ignore handled characters
		beginCopy += ignore;
		i += ignore;

	    } else {
		if (c == '\r' || c == '\n') {
		    // copy the line without CR/LF
		    toAppendTo.append(buffer,  beginCopy, i-beginCopy);

		    if (i+1 < bufferLen && c == '\r' && buffer[i+1] == '\n') {
			i += 2; // CRLF
		    } else {
			i++; // CR or LF
		    }

		    // mark the start of the line
		    lineStart = true;
		    beginCopy = i;

		    // insert line break if not in a list
		    if (state == STATE_NORMAL && (mods&FORMAT_BR) != 0) {
			toAppendTo.append(tagBr);
		    }
		} else {
		    // just go on
		    i++;
		}
	    }
	}

	// copy rest (if no crlf at end)
	if (beginCopy<bufferLen) {
	    toAppendTo.append(buffer,  beginCopy, bufferLen-beginCopy);
	}

	// close open lists
	if (state == STATE_UL) {
	    toAppendTo.append(tagUlItemClose);
	    toAppendTo.append(tagUlClose);
	    state = STATE_NORMAL;
	} else if (state == STATE_OL) {
	    toAppendTo.append(tagOlItemClose);
	    toAppendTo.append(tagOlClose);
	    state = STATE_NORMAL;
	}

	return toAppendTo;
    }

}
