/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.text;

/** String abbreviator that uses a macosx-style ellipsis algorithm, 
 *  where an ellipsis is added in the middle of a string instead of 
 *  at the end.
 */
public class StringAbbreviator {
    public static final String SPACE = " ";
    public static final String DEFAULT_ELLIPSIS = "...";
    private String ellipsis = DEFAULT_ELLIPSIS;

    /** Abbreviate str if needed, so that its number of characters
     *  is <= maxChars */
    public String abbreviate(String str, int maxLength) {
        if(str == null) {
            return null;
        }
        
        if(str.length() <= maxLength) {
            return str;
        }
        
        if(maxLength < ellipsis.length()) {
            return ellipsis.substring(0, maxLength);
        }
        
        // Take one word from the start and end of the string,
        // repeatedly, until we're over the allowed length.
        // when that happens, truncate the last word that we
        // took from the beginning
        int charsLeft = maxLength - ellipsis.length();
        final String [] words = str.trim().split(" ");
        final StringBuilder begin = new StringBuilder();
        final StringBuilder end = new StringBuilder();
        
        if(words.length == 1) {
            addUpTo(begin, words[0], true, charsLeft);
        } else {
            int i=0;
            int j=words.length - 1;
            while(i <= j && charsLeft > 0) {
                
                // Beginning: add space if needed
                if(begin.length() > 0) {
                    charsLeft = addUpTo(begin, SPACE, true, charsLeft);
                    if(charsLeft <= 0) {
                        break;
                    }
                }
                
                // And add next word from the beginning
                charsLeft = addUpTo(begin, words[i], true, charsLeft);
                if(charsLeft <= 0) {
                    break;
                }
                
                // If i==j we'd add the same word again so stop
                if(i==j) {
                    break;
                }

                // End: add space if needed
                if(end.length() > 0) {
                    charsLeft = addUpTo(end, SPACE, false, charsLeft);
                    if(charsLeft <= 0) {
                        break;
                    }
                }
                
                // End: add next word
                charsLeft = addUpTo(end, words[j], false, charsLeft);
                if(charsLeft <= 0) {
                    break;
                }
                
                i++;
                j--;
            }
        }
        
        final StringBuilder sb = new StringBuilder();
        sb.append(begin.toString());
        sb.append(ellipsis);
        sb.append(end.toString());
        return sb.toString();
    }
    
    /** Set our ellipsis string, to the default if s is null */
    public void setEllipsis(String s) {
        ellipsis = s == null ? DEFAULT_ELLIPSIS : s;
    }
    
    public String getEllipsis() {
        return ellipsis;
    }
    
    private static int addUpTo(StringBuilder sb, String str, boolean appendAtEnd, int charsLeft) {
        if(str.length() > charsLeft) {
            if(appendAtEnd) {
                str = str.substring(0, charsLeft);
            } else {
                str = str.substring(str.length() - charsLeft, str.length());
            }
        }
        if(appendAtEnd) {
            sb.append(str);
        } else {
            sb.insert(0, str);
        }
        return charsLeft - str.length();
    }
}
