/*
 * $Id: JavaVersion.java 12799 2004-11-19 16:34:44Z fmeschbe $
 *
 * Copyright 1997-2004 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.util;

/**
 * The <code>JavaVersion</code> class is a helper class to handle various
 * Java version strings and query about the version of the Java VM the
 * application is running.
 * <p>
 * The version information provided is based on the Java Runtime of the currently
 * running Virtual Machine.
 * <p>
 * This is a utility class, which cannot be instantiated.
 *
 * @author fmeschbe
 * @version $Revision: 1.4 $, $Date: 2004-11-19 17:34:44 +0100 (Fri, 19 Nov 2004) $
 * @audience wad
 * @since hawk
 */
public class JavaVersion {

    /**
     * This object is used to check for specification version compliance. The
     * nice thing about the Package class is, that it contains a method to
     * correctly check version names like 1.3.1, etc.
     */
    private static final Package specPackage;

    static {
        // Get the package from the class loader(s)
        // See bug 10745 for details.
        specPackage = new ClassLoader() {
            Package createPackage() {
                return definePackage("dummy.package", 
                    System.getProperty("java.specification.name"), 
                    System.getProperty("java.specification.version"), 
                    System.getProperty("java.specification.vendor"),
                    "Java Runtime Environment", 
                    System.getProperty("java.version"), 
                    System.getProperty("java.vendor"), null);
            }
        }.createPackage();
    }

    /** The specification version String for Java 1.3 */
    private static final String specVersion13 = "1.3";

    /** The specification version String for Java 1.4 */
    private static final String specVersion14 = "1.4";

    /** The specification version String for Java 1.5 */
    private static final String specVersion15 = "1.5";

    /** Private constructor prevents instantion */
    private JavaVersion() {}

    /**
     * Returns the implementation version of the current Java Runtime library.
     * E.g. "1.4.2_04"
     */
    public static String getVersion() {
        return specPackage.getImplementationVersion();
    }

    /**
     * Returns the specification version of the current Java Runtime library.
     * E.g. "1.4"
     */
    public static String getSpecificationVersion() {
        return specPackage.getSpecificationVersion();
    }

    /**
     * Returns <code>true</code> if the specification version of the current
     * Java Runtime library is <code>1.3</code>. This is a convenience method
     * completely equivalent to
     * {@link #isSpecificationVersion(java.lang.String) isSpecificationVersion("1.3")}.
     */
    public static boolean is13() {
        return isSpecificationVersion(specVersion13);
    }

    /**
     * Returns <code>true</code> if the specification version of the current
     * Java Runtime library is <code>1.4</code>. This is a convenience method
     * completely equivalent to
     * {@link #isSpecificationVersion(java.lang.String) isSpecificationVersion("1.4")}.
     */
    public static boolean is14() {
        return isSpecificationVersion(specVersion14);
    }

    /**
     * Returns <code>true</code> if the specification version of the current
     * Java Runtime library is <code>1.5</code>. This is a convenience method
     * completely equivalent to
     * {@link #isSpecificationVersion(java.lang.String) isSpecificationVersion("1.5")}.
     */
    public static boolean is15() {
        return isSpecificationVersion(specVersion15);
    }

    /**
     * Returns <code>true</code> if the specification version of the current
     * Java Runtime library the given version string.
     *
     * @param specVersion The specification version to compare to the
     *      specification of the current Hava Runtime library.
     */
    public static boolean isSpecificationVersion(String specVersion) {
        if (specVersion == null || specVersion.length() == 0) {
            return false;
        } else {
            return specVersion.equals(getSpecificationVersion());
        }
    }


    /**
     * Returns <code>true</code> if the specification version of the current
     * Java Runtime library is compatible with <code>1.3</code>, that is if the
     * specification version of the current Java Runtime library is the same or
     * newer than <code>1.3</code>. This is a convenience method completely
     * equivalent to
     * {@link #isCompatibleWith(java.lang.String) isCompatibleWith("1.3")}.
     */
    public static boolean isCompatibleWith13() {
        return isCompatibleWith(specVersion13);
    }

    /**
     * Returns <code>true</code> if the specification version of the current
     * Java Runtime library is compatible with <code>1.4</code>, that is if the
     * specification version of the current Java Runtime library is the same or
     * newer than <code>1.4</code>. This is a convenience method completely
     * equivalent to
     * {@link #isCompatibleWith(java.lang.String) isCompatibleWith("1.4")}.
     */
    public static boolean isCompatibleWith14() {
        return isCompatibleWith(specVersion14);
    }

    /**
     * Returns <code>true</code> if the specification version of the current
     * Java Runtime library is compatible with <code>1.5</code>, that is if the
     * specification version of the current Java Runtime library is the same or
     * newer than <code>1.5</code>. This is a convenience method completely
     * equivalent to
     * {@link #isCompatibleWith(java.lang.String) isCompatibleWith("1.5")}.
     */
    public static boolean isCompatibleWith15() {
        return isCompatibleWith(specVersion15);
    }

    /**
     * Returns <code>true</code> if the specification version of the current
     * Java Runtime library is compatible with the given specification version,
     * that is if the specification version of the current Java Runtime library
     * is the same or newer than the given version.
     *
     * @param specVersion The specification version to compare to the
     *      specification of the current Hava Runtime library.
     */
    public static boolean isCompatibleWith(String specVersion) {
        return specPackage.isCompatibleWith(specVersion);
    }

}
