/*
 * $Id: OrderedStringKeyMap.java 12345 2004-08-22 04:56:09Z fielding $
 *
 * Copyright 1997-2004 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.util;

import java.util.*;

/**
 * a simple implementation for a map with internal ordering. it actually
 * combines a list with a map. but takes only strings as keys.
 *
 * @version $Revision: 1.22 $, $Date: 2004-08-22 06:56:09 +0200 (Sun, 22 Aug 2004) $
 * @author tripod
 * @since antbear
 * @audience wad
 */
public class OrderedStringKeyMap  {

    /** the internal list */
    private final ArrayList list = new ArrayList();

    /** the internal map */
    private final HashMap map = new HashMap();

    /**
     * creates a new OrderedMap
     */
    public OrderedStringKeyMap() {
    }

    /**
     * Returns the number of the elelemts in this map
     * @return the number of the elelemts in this map
     */
    public int size() {
	return list.size();
    }

    /**
     * Checks if this map is empty.
     *
     * @return <code>true</code> if this map is empty;
     *         <code>false</code> otherwise.
     */
    public boolean isEmpty() {
	return list.isEmpty();
    }

    /**
     * Checks if this map contains the specified value.
     *
     * @param o the object the check
     *
     * @return <code>true</code> if this map contains the value;
     *         <code>false</code> otherwise.
     */
    public boolean contains(Object o) {
	return map.containsValue(o);
    }

    /**
     * Checks if this map contains the specified key.
     *
     * @param key the key to check
     *
     * @return <code>true</code> if this map contaisn the key;
     *         <code>false</code> otherwise.
     */
    public boolean contains(String key) {
	return map.containsKey(key);
    }

    /**
     * Checks if this map contains the specified value.
     *
     * @param o the object the check
     *
     * @return <code>true</code> if this map contains the value;
     *         <code>false</code> otherwise.
     */
    public boolean containsValue(Object o) {
	return map.containsValue(o);
    }

    /**
     * Checks if this map contains the specified key.
     *
     * @param key the key to check
     *
     * @return <code>true</code> if this map contaisn the key;
     *         <code>false</code> otherwise.
     */
    public boolean containsKey(String key) {
	return map.containsKey(key);
    }

    /**
     * Returns the value with the given <code>key</code> or <code>null</code> if
     * the values is not in this map.
     *
     * @param key the key of the value
     *
     * @return the value or <code>null</code>
     */
    public Object get(String key) {
	return map.get(key);
    }

    /**
     * Returns the element at the specified position in this list.
     *
     * @param  index index of element to return.
     * @return the element at the specified position in this list.
     * @throws    IndexOutOfBoundsException if index is out of range <tt>(index
     * 		  &lt; 0 || index &gt;= size())</tt>.
     */
    public Object get(int index) {
	return list.get(index);
    }

    /**
     * Returns an iterator over the elements in this list in proper
     * sequence.
     *
     * @return an iterator over the elements in this list in proper sequence.
     */
    public Iterator iterator() {
	return list.iterator();
    }

    /**
     * Returns an iterator over the elements in this list in proper
     * sequence.
     *
     * @param reverse if set to <code>true</code>, the elemens are iterated in
     *         reverse order
     *
     * @return an iterator over the elements in this list in proper sequence.
     */
    public Iterator iterator(boolean reverse) {
	return reverse ? new ReverseListIterator(list) : list.iterator();
    }

    /**
     * Returns an iterator over all keys in the map in any order.
     *
     * @return an iterator over all keys in the map in any order.
     */
    public Iterator keys() {
        return map.keySet().iterator();
    }

    /**
     * Returns an array containing all of the elements in this list
     * in the correct order.
     *
     * @return an array containing all of the elements in this list
     * 	       in the correct order.
     */
    public Object[] toArray() {
	return list.toArray();
    }


    /**
     * Returns an array containing all of the elements in this list in the
     * correct order.  The runtime type of the returned array is that of the
     * specified array.  If the list fits in the specified array, it is
     * returned therein.  Otherwise, a new array is allocated with the runtime
     * type of the specified array and the size of this list.<p>
     *
     * If the list fits in the specified array with room to spare (i.e., the
     * array has more elements than the list), the element in the array
     * immediately following the end of the collection is set to
     * <tt>null</tt>.  This is useful in determining the length of the list
     * <i>only</i> if the caller knows that the list does not contain any
     * <tt>null</tt> elements.
     *
     * @param a the array into which the elements of the list are to
     *		be stored, if it is big enough; otherwise, a new array of the
     * 		same runtime type is allocated for this purpose.
     * @return an array containing the elements of the list.
     * @throws ArrayStoreException if the runtime type of a is not a supertype
     *         of the runtime type of every element in this list.
     */
    public Object[] toArray(Object a[]) {
	return list.toArray(a);
    }

    /**
     * Removes all of the elements from this key map.  The key map will
     * be empty after this call returns.
     */
    public void clear() {
	list.clear();
	map.clear();
    }

    /**
     * Puts an element into the map and appends it to the end of the list.
     * @param key the key of the element
     * @param value the value of the element
     *
     * @return the old element that was assigned with this key or
     *         <code>null</code> if there was no such element.
     */
    public Object put(String key, Object value) {
	Object old=map.put(key, value);
	if (old!=null) list.remove(old);
	list.add(value);
	return old;
    }

    /**
     * Puts an element in this key map and replaces it in the internal list
     * if an element with the key already exists, or appends it to the end
     * of the list, otherwise.
     *
     * @param key the key of the element
     * @param value the value of the element
     *
     * @return the old element that was assigned with this key or
     *         <code>null</code> if there was no such element.
     */
    public Object set(String key, Object value) {
	Object old=map.put(key, value);
	if (old!=null) {
	    int idx = list.indexOf(old);
	    list.remove(old);
	    list.add(idx, value);
	} else {
	    list.add(value);
	}
	return old;
    }

    /**
     * Searches for the first occurence of the given argument, testing
     * for equality using the <tt>equals</tt> method.
     *
     * @param   value an object.
     * @return  the index of the first occurrence of the argument in this
     *          list; returns <tt>-1</tt> if the object is not found.
     * @see     Object#equals(Object)
     */
    public int indexOf(Object value) {
	return list.indexOf(value);
    }

    /**
     * Inserts the specified element at the specified position in this
     * list. Shifts the element currently at that position (if any) and
     * any subsequent elements to the right (adds one to their indices).
     *
     * @param key the key of the element
     * @param value element to be inserted.
     * @param index index at which the specified element is to be inserted.
     *
     * @return the old element that was assigned with this key or
     *         <code>null</code> if there was no such element.
     *
     * @throws    IndexOutOfBoundsException if index is out of range
     *		  <tt>(index &lt; 0 || index &gt; size())</tt>.
     */
    public Object put(String key, Object value, int index) {
	Object old = map.remove(key);
	if (old!=null) list.remove(old);
	list.add(index, value);
	map.put(key,value);
	return old;
    }

    /**
     * Inserts the specified element above the element specified with
     * <code>above</code>.
     *
     * @param key the key of the element
     * @param value element to be inserted.
     * @param above the key of the element to insert above
     *
     * @return the old element that was assigned with this key or
     *         <code>null</code> if there was no such element.
     */
    public Object put(String key, Object value, String above) {
	Object old=map.remove(key);
	if (old!=null) list.remove(old);
	int index=above==null?-1:list.indexOf(map.get(above));
	map.put(key, value);
	if (index<0) {
	    list.add(value);
	} else {
	    list.add(index, value);
	}
	return old;
    }

    /**
     * Removes the mapping for this key from this map if present.
     *
     * @param key key whose mapping is to be removed from the map.
     * @return previous value associated with specified key, or <tt>null</tt>
     *	       if there was no mapping for key.  A <tt>null</tt> return can
     *	       also indicate that the map previously associated <tt>null</tt>
     *	       with the specified key.
     */
    public Object remove(String key) {
	Object old=map.remove(key);
	if (old!=null) list.remove(old);
	return old;
    }

    /**
     * Removes the element at the specified position in this list.
     * Shifts any subsequent elements to the left (subtracts one from their
     * indices).
     *
     * @param index the index of the element to removed.
     * @return the element that was removed from the list.
     * @throws    IndexOutOfBoundsException if index out of range <tt>(index
     * 		  &lt; 0 || index &gt;= size())</tt>.
     */
    public Object remove(int index) {
	Object old=list.remove(index);
	if (old!=null) map.remove(old);
	return old;
    }

    /**
     * Returns the last element of this list, or <code>null</code> if this list
     * is empty.
     *
     * @return the last element of this list.
     */
    public Object lastElement() {
	if (list.isEmpty()) return null;
	return list.get(list.size()-1);
    }

    /**
     * Moves the element with the <code>key</code> above the one specified
     * with <code>above</code>.
     */
    public void move(String key, String above) {
	Object elem=map.get(key);
	if (elem==null) {
	    throw new NoSuchElementException(key);
	}
	if (key.equals(above)) {
	    return;
	}
        Object aboveElem;
	if (above==null || above.equals("") || (aboveElem=map.get(above))==null) {
	    list.remove(elem);
	    list.add(elem);
	} else {
	    list.remove(elem);
	    list.add(list.indexOf(aboveElem), elem);
	}
    }

    /**
     * Returns a collection view of the mappings contained in this map.  Each
     * element in the returned collection is a <tt>Map.Entry</tt>.  The
     * collection is backed by the map, so changes to the map are reflected in
     * the collection, and vice-versa.  The collection supports element
     * removal, which removes the corresponding mapping from the map, via the
     * <tt>Iterator.remove</tt>, <tt>Collection.remove</tt>,
     * <tt>removeAll</tt>, <tt>retainAll</tt>, and <tt>clear</tt> operations.
     * It does not support the <tt>add</tt> or <tt>addAll</tt> operations.
     *
     * @return a collection view of the mappings contained in this map.
     * @see Map.Entry
     */
    public Set entrySet() {
	return map.entrySet();
    }

    /**
     * Returns a set view of the keys contained in this map.  The set is
     * backed by the map, so changes to the map are reflected in the set, and
     * vice-versa.  The set supports element removal, which removes the
     * corresponding mapping from this map, via the <tt>Iterator.remove</tt>,
     * <tt>Set.remove</tt>, <tt>removeAll</tt>, <tt>retainAll</tt>, and
     * <tt>clear</tt> operations.  It does not support the <tt>add</tt> or
     * <tt>addAll</tt> operations.
     *
     * @return a set view of the keys contained in this map.
     */
    public Set keySet() {
	return map.keySet();
    }

    //--------------------------------------------------------------------------
    private static final class ReverseListIterator implements Iterator {

	/** the list to iterate */
	private final List list;

	/** the position of the next element to return */
	private int pos;

	/**
	 * Constructs a new <code>ReverseListIterator</code>
	 *
	 * @param list the list to iterate
	 */
	public ReverseListIterator(List list) {
	    this.list = list;
	    pos = list.size()-1;
	}

	/**
	 * @see Iterator#hasNext()
	 */
	public boolean hasNext() {
	    return pos>=0;
	}

	/**
	 * @see Iterator#next()
	 */
	public Object next() throws NoSuchElementException {
	    if (pos<0) {
		throw new NoSuchElementException();
	    }
	    return list.get(pos--);
	}

	/**
	 * @see Iterator#remove()
	 */
	public void remove() {
	    throw new UnsupportedOperationException();
	}
    }
}