/*
 * $URL$
 * $Id$
 *
 * Copyright 1997-2006 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.util.diff;

import java.io.PrintWriter;

/**
 * Provides a default output for a diff.
 *
 * @author tripod
 * @version $Rev: 26301 $, $Date: 2007-04-26 18:55:18 +0200 (Thu, 26 Apr 2007) $
 */
public class DefaultChangeListener implements ChangeListener {

    /**
     * the output writer
     */
    private final PrintWriter out;

    /**
     * debug flag
     */
    private boolean debug;

    /**
     * Creates a new default change listener that will write to the given
     * writer.
     * @param out the writer
     */
    public DefaultChangeListener(PrintWriter out) {
        this.out = out;
    }

    /**
     * Creates a new default change listener that will write to the given
     * writer. if debug is <code>true</code> the line numbers are also included
     * in the output.
     *
     * @param out the writer
     * @param debug flag
     */
    public DefaultChangeListener(PrintWriter out, boolean debug) {
        this.out = out;
        this.debug = debug;
    }

    /**
     * {@inheritDoc}
     */
    public void onDocumentsStart(Document left, Document right) {
        out.println("Start Diff");
    }

    /**
     * {@inheritDoc}
     */
    public void onDocumentsEnd(Document left, Document right) {
        out.println("End Diff");
    }

    /**
     * {@inheritDoc}
     */
    public void onChangeStart(int leftLine, int leftLen, int rightLine, int rightLen) {
        out.println("@@ -" + (leftLine+1) + "," + leftLen + " +" + (rightLine+1) + "," + rightLen + " @@");
    }

    /**
     * {@inheritDoc}
     */
    public void onChangeEnd() {
        out.flush();
    }

    /**
     * {@inheritDoc}
     */
    public void onUnmodified(int leftLine, int rightLine, Document.Element text) {
        if (debug) {
            out.print("(" + (leftLine+1) + "," + (rightLine+1) + ") ");
        }
        out.println(text);
    }

    /**
     * {@inheritDoc}
     */
    public void onDeleted(int leftLine, int rightLine, Document.Element text) {
        if (debug) {
            out.print("(" + (leftLine+1) + "," + (rightLine+1) + ") ");
        }
        out.print("-");
        out.println(text);
    }

    /**
     * {@inheritDoc}
     */
    public void onInserted(int leftLine, int rightLine, Document.Element text) {
        if (debug) {
            out.print("(" + (leftLine+1) + "," + (rightLine+1) + ") ");
        }
        out.print("+");
        out.println(text);
    }

}
