/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.blob.datastore;

import com.google.common.base.Charsets;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.io.Files;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.jackrabbit.core.data.DataRecord;
import org.apache.jackrabbit.oak.commons.FileIOUtils;
import org.apache.jackrabbit.oak.commons.concurrent.ExecutorCloser;
import org.apache.jackrabbit.oak.plugins.blob.SharedDataStore;
import org.apache.jackrabbit.oak.plugins.blob.datastore.BlobTracker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlobIdTracker
implements Closeable,
BlobTracker {
    private static final Logger LOG = LoggerFactory.getLogger(BlobIdTracker.class);
    private final boolean SKIP_TRACKER = Boolean.getBoolean("oak.datastore.skipTracker");
    private static final String datastoreMeta = "blobids";
    private static final String fileNamePrefix = "blob";
    private static final String mergedFileSuffix = ".refs";
    private final String instanceId = UUID.randomUUID().toString();
    private final SharedDataStore datastore;
    protected BlobIdStore store;
    private final ScheduledExecutorService scheduler;
    private String prefix;

    public BlobIdTracker(String path, String repositoryId, long snapshotIntervalSecs, SharedDataStore datastore) throws IOException {
        this(path, repositoryId, Executors.newSingleThreadScheduledExecutor(), snapshotIntervalSecs, snapshotIntervalSecs, datastore);
    }

    public BlobIdTracker(String path, String repositoryId, ScheduledExecutorService scheduler, long snapshotDelaySecs, long snapshotIntervalSecs, SharedDataStore datastore) throws IOException {
        String root = FilenameUtils.concat((String)path, (String)datastoreMeta);
        File rootDir = new File(root);
        this.datastore = datastore;
        this.scheduler = scheduler;
        try {
            FileUtils.forceMkdir((File)rootDir);
            this.prefix = "blob-" + repositoryId;
            this.store = new BlobIdStore(rootDir, this.prefix);
            scheduler.scheduleAtFixedRate(new SnapshotJob(), TimeUnit.SECONDS.toMillis(snapshotDelaySecs), TimeUnit.SECONDS.toMillis(snapshotIntervalSecs), TimeUnit.MILLISECONDS);
        }
        catch (IOException e) {
            LOG.error("Error initializing blob tracker", (Throwable)e);
            this.close();
            throw e;
        }
    }

    @Override
    public void remove(File recs) throws IOException {
        this.store.removeRecords(recs);
    }

    @Override
    public void remove(Iterator<String> recs) throws IOException {
        this.store.removeRecords(recs);
    }

    @Override
    public void add(String id) throws IOException {
        if (!this.SKIP_TRACKER) {
            this.store.addRecord(id);
        }
    }

    @Override
    public void add(Iterator<String> recs) throws IOException {
        if (!this.SKIP_TRACKER) {
            this.store.addRecords(recs);
        }
    }

    @Override
    public void add(File recs) throws IOException {
        if (!this.SKIP_TRACKER) {
            this.store.addRecords(recs);
        }
    }

    @Override
    public Iterator<String> get() throws IOException {
        try {
            if (!this.SKIP_TRACKER) {
                this.globalMerge();
                return this.store.getRecords();
            }
            return Iterators.emptyIterator();
        }
        catch (IOException e) {
            LOG.error("Error in retrieving blob records iterator", (Throwable)e);
            throw e;
        }
    }

    @Override
    public File get(String path) throws IOException {
        if (!this.SKIP_TRACKER) {
            this.globalMerge();
            return this.store.getRecords(path);
        }
        return new File(path);
    }

    private void globalMerge() throws IOException {
        try {
            Stopwatch watch = Stopwatch.createStarted();
            LOG.trace("Retrieving all blob id files available form the DataStore");
            List<DataRecord> refRecords = this.datastore.getAllMetadataRecords(fileNamePrefix);
            ArrayList<File> refFiles = Lists.newArrayList(Iterables.transform(refRecords, new Function<DataRecord, File>(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public File apply(DataRecord input) {
                    InputStream inputStream = null;
                    try {
                        inputStream = input.getStream();
                        File file = FileIOUtils.copy(inputStream);
                        return file;
                    }
                    catch (Exception e) {
                        LOG.warn("Error copying data store file locally {}", (Object)input.getIdentifier(), (Object)e);
                    }
                    finally {
                        IOUtils.closeQuietly((InputStream)inputStream);
                    }
                    return null;
                }
            }));
            LOG.info("Retrieved all blob id files in [{}]", (Object)watch.elapsed(TimeUnit.MILLISECONDS));
            watch = Stopwatch.createStarted();
            this.store.merge(refFiles, true);
            LOG.info("Merged all retrieved blob id files in [{}]", (Object)watch.elapsed(TimeUnit.MILLISECONDS));
            watch = Stopwatch.createStarted();
            for (DataRecord rec : refRecords) {
                this.datastore.deleteMetadataRecord(rec.getIdentifier().toString());
                LOG.debug("Deleted metadata record {}", (Object)rec.getIdentifier().toString());
            }
            LOG.info("Deleted all blob id metadata files in [{}]", (Object)watch.elapsed(TimeUnit.MILLISECONDS));
        }
        catch (IOException e) {
            LOG.error("Error in merging blob records iterator from the data store", (Throwable)e);
            throw e;
        }
    }

    private void snapshot() throws IOException {
        try {
            if (!this.SKIP_TRACKER) {
                Stopwatch watch = Stopwatch.createStarted();
                this.store.snapshot();
                LOG.debug("Completed snapshot in [{}]", (Object)watch.elapsed(TimeUnit.MILLISECONDS));
                watch = Stopwatch.createStarted();
                this.datastore.addMetadataRecord(this.store.getBlobRecordsFile(), this.prefix + this.instanceId + mergedFileSuffix);
                LOG.info("Added blob id metadata record in DataStore in [{}]", (Object)watch.elapsed(TimeUnit.MILLISECONDS));
            }
        }
        catch (Exception e) {
            LOG.error("Error taking snapshot", (Throwable)e);
            throw new IOException("Snapshot error", e);
        }
    }

    @Override
    public void close() throws IOException {
        this.store.close();
        new ExecutorCloser(this.scheduler).close();
    }

    class SnapshotJob
    implements Runnable {
        SnapshotJob() {
        }

        @Override
        public void run() {
            try {
                BlobIdTracker.this.snapshot();
                LOG.info("Finished taking snapshot");
            }
            catch (Exception e) {
                LOG.warn("Failure in taking snapshot", (Throwable)e);
            }
        }
    }

    static class BlobIdStore
    implements Closeable {
        private static final String genFileNameSuffix = ".gen";
        private static final String workingCopySuffix = ".process";
        private BufferedWriter writer;
        private File processFile;
        private final List<File> generations;
        private final File rootDir;
        private final String prefix;
        private final ReentrantLock refLock;
        private final ReentrantLock snapshotLock;

        BlobIdStore(File rootDir, String prefix) throws IOException {
            this.rootDir = rootDir;
            this.prefix = prefix;
            this.refLock = new ReentrantLock();
            this.snapshotLock = new ReentrantLock();
            this.processFile = Files.fileTreeTraverser().breadthFirstTraversal(rootDir).firstMatch(Type.IN_PROCESS.filter()).orNull();
            this.generations = Collections.synchronizedList(Lists.newArrayList(Files.fileTreeTraverser().breadthFirstTraversal(rootDir).filter(Type.GENERATION.filter())));
            this.nextGeneration();
        }

        protected synchronized void addRecord(String id) throws IOException {
            this.writer.append(id);
            this.writer.newLine();
            this.writer.flush();
            LOG.debug("Added record {}", (Object)id);
        }

        protected Iterator<String> getRecords() throws IOException {
            try {
                String path = File.createTempFile("temp", null).getAbsolutePath();
                return FileIOUtils.BurnOnCloseFileIterator.wrap(FileUtils.lineIterator((File)this.getRecords(path)), new File(path));
            }
            catch (IOException e) {
                LOG.error("Error in retrieving blob records iterator", (Throwable)e);
                throw e;
            }
        }

        protected File getRecords(String path) throws IOException {
            this.refLock.lock();
            File copiedRecsFile = new File(path);
            try {
                FileUtils.copyFile((File)this.getBlobRecordsFile(), (File)copiedRecsFile);
                File file = copiedRecsFile;
                return file;
            }
            catch (IOException e) {
                LOG.error("Error in retrieving blob records file", (Throwable)e);
                throw e;
            }
            finally {
                this.refLock.unlock();
            }
        }

        protected File getBlobRecordsFile() throws IOException {
            File refs = new File(this.rootDir, this.prefix + Type.REFS.getFileNameSuffix());
            if (!refs.exists()) {
                LOG.debug("File created {}", (Object)refs.createNewFile());
            }
            return refs;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void merge(List<File> refFiles, boolean doSort) throws IOException {
            this.refLock.lock();
            try {
                if (refFiles != null && !refFiles.isEmpty()) {
                    File merged = new File(this.rootDir, this.prefix + Type.REFS.getFileNameSuffix());
                    FileIOUtils.append(refFiles, merged, true);
                    LOG.debug("Merged files into references {}", refFiles);
                    refFiles.clear();
                }
                if (doSort) {
                    FileIOUtils.sort(this.getBlobRecordsFile());
                }
            }
            finally {
                this.refLock.unlock();
            }
        }

        protected void removeRecords(Iterator<String> recs) throws IOException {
            File deleted = File.createTempFile("deleted", null);
            FileIOUtils.writeStrings(recs, deleted, false);
            this.removeRecords(deleted);
            LOG.trace("Removed records");
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void removeRecords(File recs) throws IOException {
            this.snapshot();
            this.refLock.lock();
            try {
                FileIOUtils.sort(this.getBlobRecordsFile());
                FileIOUtils.sort(recs);
                LOG.trace("Sorted files");
                File temp = File.createTempFile("sorted", null);
                try (FileIOUtils.FileLineDifferenceIterator iterator = null;){
                    iterator = new FileIOUtils.FileLineDifferenceIterator(recs, this.getBlobRecordsFile(), null);
                    FileIOUtils.writeStrings(iterator, temp, false);
                }
                File blobRecs = this.getBlobRecordsFile();
                Files.move(temp, blobRecs);
                LOG.trace("removed records");
            }
            finally {
                this.refLock.unlock();
                try {
                    FileUtils.forceDelete((File)recs);
                }
                catch (IOException e) {
                    LOG.debug("Failed to delete file {}", (Object)recs, (Object)e);
                }
            }
        }

        private synchronized void nextGeneration() throws IOException {
            this.close();
            this.processFile = new File(this.rootDir, this.prefix + Type.IN_PROCESS.getFileNameSuffix());
            this.writer = Files.newWriter(this.processFile, Charsets.UTF_8);
            LOG.info("Created new process file and writer over {} ", (Object)this.processFile.getAbsolutePath());
        }

        protected void addRecords(Iterator<String> recs) throws IOException {
            File added = File.createTempFile("added", null);
            FileIOUtils.writeStrings(recs, added, false);
            this.merge(Lists.newArrayList(added), false);
        }

        protected void addRecords(File recs) throws IOException {
            this.merge(Lists.newArrayList(recs), false);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void snapshot() throws IOException {
            this.snapshotLock.lock();
            try {
                this.nextGeneration();
                this.merge(this.generations, false);
            }
            finally {
                this.snapshotLock.unlock();
            }
        }

        @Override
        public synchronized void close() {
            IOUtils.closeQuietly((Writer)this.writer);
            LOG.info("Closed writer");
            if (this.processFile != null) {
                File renamed = new File(FilenameUtils.removeExtension((String)this.processFile.getAbsolutePath()));
                boolean success = this.processFile.renameTo(renamed);
                LOG.debug("File renamed {}", (Object)success);
                if (success) {
                    this.generations.add(renamed);
                    LOG.info("Process file renamed to {}", (Object)renamed.getAbsolutePath());
                } else {
                    LOG.trace("Trying a copy file operation");
                    try {
                        if (renamed.createNewFile()) {
                            Files.copy(this.processFile, renamed);
                            this.generations.add(renamed);
                            LOG.info("{} File copied to {}", (Object)this.processFile.getAbsolutePath(), (Object)renamed.getAbsolutePath());
                        }
                    }
                    catch (Exception e) {
                        LOG.warn("Unable to copy process file to corresponding gen file. Some elements may be missed", (Throwable)e);
                    }
                }
            }
        }

        static enum Type {
            IN_PROCESS{

                @Override
                String getFileNameSuffix() {
                    return "." + System.currentTimeMillis() + BlobIdStore.genFileNameSuffix + BlobIdStore.workingCopySuffix;
                }

                @Override
                Predicate<File> filter() {
                    return new Predicate<File>(){

                        @Override
                        public boolean apply(File input) {
                            return input.getName().endsWith(BlobIdStore.workingCopySuffix) && input.getName().startsWith(BlobIdTracker.fileNamePrefix);
                        }
                    };
                }
            }
            ,
            GENERATION{

                @Override
                String getFileNameSuffix() {
                    return "." + System.currentTimeMillis() + BlobIdStore.genFileNameSuffix;
                }

                @Override
                Predicate<File> filter() {
                    return new Predicate<File>(){

                        @Override
                        public boolean apply(File input) {
                            return input.getName().startsWith(BlobIdTracker.fileNamePrefix) && input.getName().contains(BlobIdStore.genFileNameSuffix) && !input.getName().endsWith(BlobIdStore.workingCopySuffix);
                        }
                    };
                }
            }
            ,
            REFS{

                @Override
                String getFileNameSuffix() {
                    return BlobIdTracker.mergedFileSuffix;
                }

                @Override
                Predicate<File> filter() {
                    return new Predicate<File>(){

                        @Override
                        public boolean apply(File input) {
                            return input.getName().endsWith(BlobIdTracker.mergedFileSuffix) && input.getName().startsWith(BlobIdTracker.fileNamePrefix);
                        }
                    };
                }
            };


            String getFileNameSuffix() {
                return "";
            }

            Predicate<File> filter() {
                return Predicates.alwaysTrue();
            }
        }
    }
}

