/*******************************************************************************
 *
 *    Copyright 2020 Adobe. All rights reserved.
 *    This file is licensed to you under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License. You may obtain a copy
 *    of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software distributed under
 *    the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 *    OF ANY KIND, either express or implied. See the License for the specific language
 *    governing permissions and limitations under the License.
 *
 ******************************************************************************/

package com.adobe.cq.commerce.magento.graphql;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import com.shopify.graphql.support.AbstractQuery;
import com.shopify.graphql.support.Input;

public class BraintreeInput implements Serializable {
    private boolean isActivePaymentTokenEnabler;

    private String paymentMethodNonce;

    private Input<String> deviceData = Input.undefined();

    private Map<String, Input<Serializable>> customFilters = new HashMap<>();

    public BraintreeInput(boolean isActivePaymentTokenEnabler, String paymentMethodNonce) {
        this.isActivePaymentTokenEnabler = isActivePaymentTokenEnabler;

        this.paymentMethodNonce = paymentMethodNonce;
    }

    /**
     * States whether an entered by a customer credit/debit card should be tokenized for later usage.
     * Required only if Vault is enabled for Braintree payment integration.
     */
    public boolean getIsActivePaymentTokenEnabler() {
        return isActivePaymentTokenEnabler;
    }

    /**
     * States whether an entered by a customer credit/debit card should be tokenized for later usage.
     * Required only if Vault is enabled for Braintree payment integration.
     */
    public BraintreeInput setIsActivePaymentTokenEnabler(boolean isActivePaymentTokenEnabler) {
        this.isActivePaymentTokenEnabler = isActivePaymentTokenEnabler;
        return this;
    }

    /**
     * The one-time payment token generated by Braintree payment gateway based on card details. Required
     * field to make sale transaction.
     */
    public String getPaymentMethodNonce() {
        return paymentMethodNonce;
    }

    /**
     * The one-time payment token generated by Braintree payment gateway based on card details. Required
     * field to make sale transaction.
     */
    public BraintreeInput setPaymentMethodNonce(String paymentMethodNonce) {
        this.paymentMethodNonce = paymentMethodNonce;
        return this;
    }

    /**
     * Contains a fingerprint provided by Braintree JS SDK and should be sent with sale transaction details
     * to the Braintree payment gateway. Should be specified only in a case if Kount (advanced fraud
     * protection) is enabled for Braintree payment integration.
     */
    public String getDeviceData() {
        return deviceData.getValue();
    }

    /**
     * Contains a fingerprint provided by Braintree JS SDK and should be sent with sale transaction details
     * to the Braintree payment gateway. Should be specified only in a case if Kount (advanced fraud
     * protection) is enabled for Braintree payment integration.
     */
    public Input<String> getDeviceDataInput() {
        return deviceData;
    }

    /**
     * Contains a fingerprint provided by Braintree JS SDK and should be sent with sale transaction details
     * to the Braintree payment gateway. Should be specified only in a case if Kount (advanced fraud
     * protection) is enabled for Braintree payment integration.
     */
    public BraintreeInput setDeviceData(String deviceData) {
        this.deviceData = Input.optional(deviceData);
        return this;
    }

    /**
     * Contains a fingerprint provided by Braintree JS SDK and should be sent with sale transaction details
     * to the Braintree payment gateway. Should be specified only in a case if Kount (advanced fraud
     * protection) is enabled for Braintree payment integration.
     */
    public BraintreeInput setDeviceDataInput(Input<String> deviceData) {
        if (deviceData == null) {
            throw new IllegalArgumentException("Input can not be null");
        }
        this.deviceData = deviceData;
        return this;
    }

    /**
     * Set custom filter.
     */
    public BraintreeInput setCustomFilter(String name, Serializable filterInput) {
        this.customFilters.put(name, Input.optional(filterInput));
        return this;
    }

    public void appendTo(StringBuilder _queryBuilder) {
        String separator = "";
        _queryBuilder.append('{');

        if (!this.customFilters.isEmpty()) {
            for (Map.Entry<String, Input<Serializable>> entry : customFilters.entrySet()) {
                _queryBuilder.append(separator);
                separator = ",";
                _queryBuilder.append(entry.getKey() + ":");

                Serializable filter = entry.getValue().getValue();

                if (filter != null) {
                    try {
                        Method appendTo = filter.getClass().getMethod("appendTo", StringBuilder.class);
                        appendTo.invoke(filter, _queryBuilder);
                    } catch (NoSuchMethodException | InvocationTargetException | IllegalAccessException e) {
                        _queryBuilder.append("null");
                    }
                } else {
                    _queryBuilder.append("null");
                }
            }
        }

        _queryBuilder.append(separator);
        separator = ",";
        _queryBuilder.append("is_active_payment_token_enabler:");
        _queryBuilder.append(isActivePaymentTokenEnabler);

        _queryBuilder.append(separator);
        separator = ",";
        _queryBuilder.append("payment_method_nonce:");
        AbstractQuery.appendQuotedString(_queryBuilder, paymentMethodNonce.toString());

        if (this.deviceData.isDefined()) {
            _queryBuilder.append(separator);
            separator = ",";
            _queryBuilder.append("device_data:");
            if (deviceData.getValue() != null) {
                AbstractQuery.appendQuotedString(_queryBuilder, deviceData.getValue().toString());
            } else {
                _queryBuilder.append("null");
            }
        }

        _queryBuilder.append('}');
    }
}
