/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
/**
 * A helper class providing a set of utilities related to internationalization
 * (i18n).
 * @static
 * @singleton
 * @class CQ.I18n
 */
CQ.utils.I18n = function() {

    /**
     * The map where the dictionaries are stored under their locale.
     * @private
     * @type Object
     */
    var dicts = new Object();

    /**
     * The initialization state of the internationalization.
     * @private
     * @type Boolean
     */
    var initialized = false;

    /**
     * The URL used to request dictionaries from the server.
     * @private
     * @type String
     */
    var currentUrl = "/crx/de/dictionary.jsp";

    /**
     * The current locale.
     * @private
     * @static
     * @type String
     */
    var currentLocale = null;

    var languages = null;

    return {

        /**
         * The default locale (en).
         * @static
         * @final
         * @type String
         */
        LOCALE_DEFAULT: "en",

        /**
         * Initializes the i18n.
         * Example config:
         * <pre>{
         *   "locale": "en",
         *   "dicts": {}
         * }</pre>
         * @param {Object} config The config
         */
        init: function(config) {
            if (!config) {
                config = new Object();
            }
            if (config.locale) {
                CQ.utils.I18n.setLocale(config.locale);
            }
            if (config.url) {
                CQ.utils.I18n.setUrl(config.url);
            }
            initialized = true;
        },

        /**
         * Returns the current locale or the default locale if none is defined.
         * @static
         * @return {String} The locale
         */
        getLocale: function() {
            return currentLocale ? currentLocale : CQ.utils.I18n.LOCALE_DEFAULT;
        },

        /**
         * Sets the current locale.
         * @static
         * @param {String} locale The locale
         */
        setLocale: function(locale) {
            currentLocale = locale;
        },

        /**
         * Sets the URL used to request dictionaries from
         * the server. The locale will be appended as query string.
         * @static
         * @param {String} url The URL
         */
        setUrl: function(url) {
            currentUrl = url;
        },

        /**
         * Returns the dictionary for the specified locale. If no locale
         * is specified, the current locale is used.
         * @static
         * @param {String} locale (optional) The locale
         * @return {Object} The dictionary
         */
        getDictionary: function(locale) {
            if (!locale) {
                locale = CQ.utils.I18n.getLocale();
            }
            if (!dicts[locale]) {
                try {
                    var url = currentUrl + "?language=" + locale;
                    var response = CQ.utils.I18n.httpGet(url);
                    if (response && response.status == 200) {
                        dicts[locale] = CQ.utils.I18n.eval(response.body);
                    }
                } catch (e) {                    
                }
                if (!dicts[locale]) {
                    dicts[locale] = {};
                }
            }
            return dicts[locale];
        },

        /**
         * Translates the specified text into the current language.
         * @static
         * @param {String} text The text to translate
         * @param {String[]} snippets The snippets replacing <code>{n}</code> (optional)
         * @param {String} note A hint for translators (optional)
         * @return {String} The translated text
         */
        getMessage: function(text, snippets, note) {
            var dict, newText, lookupText;
            lookupText = note ? text + " ((" + note + "))" : text;
            if (initialized) {
                dict = CQ.utils.I18n.getDictionary();
            }
            if (dict) {
                newText = dict[lookupText];
            }
            if (!newText) {
                newText = text;
            }
            //CQ.Log.trace("CQ.utils.I18n#getMessage: translating '{0}' with '{1}'", [text, newText]);
            return CQ.utils.I18n.patchText(newText, snippets);
        },

        /**
         * Translates the specified text into the current language. Use this
         * method to translate String variables, e.g. data from the server.
         * @static
         * @param {String} text The text to translate
         * @param {String} note A hint for translators (optional)
         * @return {String} The translated text
         */
        getVarMessage: function(text, note) {
            if (!text) {
                return null;
            }
            return this.getMessage(text, null, note);
        },
        
        /**
         * Parses a language code string such as "de_CH" and returns an object with
         * language and country extracted.
         * @static
         * @param {String} langCode a language code such as "de" or "de_CH"
         * @return {Object} an object with "code" ("de_CH"), "language" ("de") and "country" ("CH")
         *                  (or null if langCode was null)
         * @since 5.4
         */
        parseLocale: function(langCode) {
            if (!langCode) {
                return null;
            }
            var pos = langCode.indexOf("_"),
                language, country;
            if (pos < 0) {
                language = langCode;
                country = null;
            } else {
                language = langCode.substring(0, pos);
                country = langCode.substring(pos + 1);
            }
            return {
                code: langCode,
                language: language,
                country: country
            };
        },

        /**
         * Replaces occurrences of <code>{n}</code> in the specified text with
         * the texts from the snippets.
         * <p>Example 1 (single snippet):<pre><code>
var text = CQ.utils.I18n.patchText("{0} has signed in.", "Jack");
           </code></pre>Result 1:<pre><code>
Jack has signed in.
           </code></pre></p>
         * <p>Example 2 (multiple snippets):<pre><code>
var text = "{0} {1} has signed in from {2}.";
text = CQ.utils.I18n.patchText(text, ["Jack", "McFarland", "10.0.0.99"]);
           </code></pre>Result 2:<pre><code>
Jack McFarland has signed in from 10.0.0.99.
           </code></pre></p>
         * @static
         * @param {String} text The text
         * @param {String/String[]} snippets The text(s) replacing
         *        <code>{n}</code>
         * @return {String} The patched text
         */
        patchText: function(text, snippets) {
            if (snippets) {
                if (!Ext.isArray(snippets)) {
                    text = text.replace("{0}", snippets);
                } else {
                    for (var i=0; i < snippets.length; i++) {
                        text = text.replace(("{" + i + "}"), snippets[i]);
                    }
                }
            }
            return text;
        },

        /**
         * Requests the specified URL from the server using GET. The request
         * will be synchronous, unless a callback function is specified.
         * @static
         * @param {String} url The URL to request
         * @param {Function} callback (optional) The callback function which is
         *        called regardless of success or failure and is passed the following
         *        parameters:<ul>
         *        <li><b>options</b> : Object<div class="sub-desc">The parameter to the request call.</div></li>
         *        <li><b>success</b> : Boolean<div class="sub-desc">True if the request succeeded.</div></li>
         *        <li><b>response</b> : Object<div class="sub-desc">The response object.</div></li>
         *        </ul>
         * @return {Mixed} The response object or, if the
         *         request is asynchronous, the transaction ID
         */
        httpGet: function(url, callback) {
            if (callback != undefined) {
                return Ext.Ajax.request({
                    url: url,
                    callback: callback
                });
            } else {
                var request = document.all ? new ActiveXObject("Microsoft.XMLHTTP") : new XMLHttpRequest();
                try {
                    request.open("GET", url, false);
                    request.send(null);
                    return {
                        status: request.status,
                        body: request.responseText
                    };
                } catch (e) {
                    return null;
                }
            }
        },

        /**
         * Evaluates and returns a string.
         * @param {String} param The string
         * @return The evaluated object
         * @type Object
         */
        eval: function(json) {
            try {
                return eval("(" + json + ")");
            } catch (e) {
                return null;
            }
        }
    };

}();

// shortcut
CQ.I18n = CQ.utils.I18n;

