/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
//------------------------------------------------------------------------------
/**
 * IFrameSet Documentation
 *
 * The IFrameSet object is best described by referring to the HTML element 
 * <frameset>. It is a container that hold a number of IFrame objects and 
 * draw them in 1 of 3 predefined layouts. Its features include resizing 
 * and memorizing of user settings. 
 *
 * Parameters:
 *
 * - id            The ID for the IFrameSet object. It must be unique
 *                 throughout the document and may be alpha-numeric, as long as
 *                 the first character is a letter. The ID is case-sensitive.
 *                 If left empty or undefined, it is generated using the
 *                 following form: iframe_[n] (where [n] is an auto-
 *                 incrementing, zero-based digit).
 *                 Note:    In order to be able to memorize user settings,
 *                          the IFrameSet object needs a specific ID.
 *
 * Required properties:
 *
 * - iFrames       An Array object holding the IFrame objects. Default value
 *                 is an empty Array. See the IFrame documentation for more
 *                 information on IFrame objects.
 *
 * Optional properties:
 *
 * - layout        Defines the layout of the IFrameSet.
 *                 The following values are accepted:
 *                     IFrameSet_LAYOUT_1 (Default)
 *                     +---+---+
 *                     | 1 | 2 |
 *                     +---+---+
 *                     IFrameSet_LAYOUT_2    IFrameSet_LAYOUT_3
 *                     +-------+             +---+---+
 *                     |   1   |             |   | 2 |
 *                     +-------+             | 1 +---+
 *                     |   2   |             |   | 3 |
 *                     +-------+             +---+---+
 *
 * - spacing       The spacing size in pixels IFrame objects should be
 *                 separated by (i.e. the size of the resizing bars). Default 
 *                 value is 10.
 *
 * - offsetWidth   The horizontal size in pixels the IFrameSet object should 
 *                 leave empty, e.g. for other objects to the left or right of
 *                 it. Default value is 0.
 *
 * - offsetHeight  The vertical size in pixels the IFrameSet object should 
 *                 leave empty, e.g. for other objects above or below. Default
 *                 value is 0.
 *
 * - minWidth      The minimum width in pixels IFrame objects can be resized 
 *                 to. Default value is 40.
 *
 * - minHeight     The minimum height in pixels IFrame objects can be resized
 *                 to. Default value is 40.
 *
 * - resizable     Enables or disables resizing of the IFrame objects. 
 *                 Accepted values are:
 *                     <code>true</code>   Resizing enabled (Default)
 *                     <code>false</code>  Resizing disabled
 *
 * - bgColor       The background color of the IFrameSet which will override
 *                 the document's background color. Default value is an empty
 *                 String.
 *
 * - gripSrc       The image displayed in the resizing areas. Note: Images will
 *                 be expanded to fill the entire area. Default value is an
 *                 empty String.
 *
 * - gripTitle     The quick info title shown when the mouse is hovering over
 *                 the resizing areas if resizing is enabled. Default value is
 *                 "Resize".
 *
 * - allowCookies  Enables or disables memorizing of user settings. 
 *                 Accepted values are:
 *                     <code>true</code>   Memory enabled (Default)
 *                     <code>false</code>  Memory disabled
 *
 * - cookieHost    The server name user settings should be memorized for if
 *                 user setting memory is enabled. Default value is an empty 
 *                 String. 
 *
 * - cookiePath    The server path user settings should be memorized for if
 *                 user setting memory is enabled. Default value is "/". 
 *
 * - cookieTTL     The number of days user settings should be memorized for if
 *                 user setting memory is enabled. Default value is 30.
 *
 * Methods:
 *
 * - draw()        Draws the IFrameSet including its IFrame objects to the
 *                 HTML document.
 */

/*
 * Static method to initiate resizing of IFrame objects.
 * @param id      The ID of the IFrameSet object to resize
 * @param axis    The resizing axis
 * @param myEvent The mouse event
 * @return        <code>true</code> if the IFrameSet object with the specified
 *                ID exists, <code>false</code> otherwise.
 */
/*boolean*/function IFrameSet_startResize(id,axis,myEvent)
	{
	var ifs = IFrameSet_get(id);
	if (!ifs) return false;
	Util.stopEventBubble(myEvent);
	Util.setEventReturnValue(myEvent,false);
	ifs.resizePosX = Util.getMousePosX(myEvent);
	ifs.resizePosY = Util.getMousePosY(myEvent);
	ifs.resizeState = IFrameSet_STATE_ON;
	ifs.resizeAxis = axis;
	ifs.showArea();
	return true;
	}

/*
 * Static method to perform resizing of IFrame objects.
 * @param id      The ID of the IFrameSet object to resize
 * @param myEvent The mouse event
 * @return        <code>true</code> if the IFrameSet object with the specified
 *                ID exists, <code>false</code> otherwise.
 */
/*boolean*/function IFrameSet_doResize(id,myEvent)
	{
	var ifs = IFrameSet_get(id);
	if (!ifs || !ifs.isResized()) return false;
	if (ifs.resizePosX >= 0 && ifs.resizePosY >= 0)
		{
		// handle resize event
		var x = Util.getMousePosX(myEvent);
		var y = Util.getMousePosY(myEvent);
		var axis = ifs.resizeAxis;
		var grip = Util.getElem(IFrameSet_PRE_GRIP+ifs.id+axis);
		if (axis == IFrameSet_HORIZONTAL)
			{
			if (ifs.resizePosX > x)
				{
				ifs.resizeWestEast(IFrameSet_LEFT,ifs.resizePosX-x,myEvent);
				}
			else if (ifs.resizePosX < x)
				{
				ifs.resizeWestEast(IFrameSet_RIGHT,x-ifs.resizePosX,myEvent);
				}
			}
		else
			{
			if (ifs.resizePosY > y)
				{
				ifs.resizeNorthSouth(IFrameSet_UP,ifs.resizePosY-y,myEvent);
				}
			else if (ifs.resizePosY < y)
				{
				ifs.resizeNorthSouth(IFrameSet_DOWN,y-ifs.resizePosY,myEvent);
				}
			}
		ifs.resizePosX = x;
		ifs.resizePosY = y;
		}
	return true;
	}


/*
 * Static method to end resizing of IFrame objects.
 * @param id      The ID of the IFrameSet object to resize
 * @param axis    The resizing axis
 * @param myEvent The mouse event
 * @return        <code>true</code> if the IFrameSet object with the specified
 *                ID exists, <code>false</code> otherwise.
 */
/*boolean*/function IFrameSet_stopResize(id,myEvent)
	{
	var ifs = IFrameSet_get(id);
	if (!ifs || !ifs.isResized()) return false;
	Util.stopEventBubble(myEvent);
	Util.setEventReturnValue(myEvent,false);
	ifs.hideArea();
	ifs.resizePosX = IFrameSet_DEFAULT_POS;
	ifs.resizePosY = IFrameSet_DEFAULT_POS;
	ifs.resizeState = IFrameSet_STATE_OFF;
	if (ifs.allowCookies)
		{
		ifs.saveUserSettings();
		}
	return true;
	}

/*
 *  Static container holding references to IFrameSet objects.
 */
IFrameSet_Static = new Object();

/*
 * Static method to retrieve the IFrameSet object with the specified ID.
 */
/*IFrameSet*/function IFrameSet_get(id)
	{
	return IFrameSet_Static[id];
	}

/*
 * Static method to store an IFrameSet object for future static use.
 */
/*void*/function IFrameSet_set(ifs)
	{
	IFrameSet_Static[ifs.id] = ifs;
	}

/*
 * Static method to prevent the selection of text in the document.
 */
/*boolean*/function IFrameSet_preventSelection()
	{
	return false;
	}

/*
 * Refreshes all IFrameSets
 */
/*void*/function IFrameSet_refresh()
    {
    for (var i in IFrameSet_Static)
        {
        IFrameSet_Static[i].refresh();
        }
    //window.location.href = window.location.href;
    }

// constants -------------------------------------------------------------------
IFrameSet_LAYOUT_1         = 2;
IFrameSet_LAYOUT_2         = 4;
IFrameSet_LAYOUT_3         = 8;
IFrameSet_DEFAULT_POS      = -1;
IFrameSet_MIN_SIZE         = 40;
IFrameSet_SPACING          = 10;
IFrameSet_OFFSET           = 0;
IFrameSet_STATE_OFF        = 0;
IFrameSet_STATE_ON         = 1;
IFrameSet_COOKIE_HOST      = "";
IFrameSet_COOKIE_PATH      = "/";
IFrameSet_COOKIE_TTL       = 30;
IFrameSet_WIDTH            = "width";
IFrameSet_HEIGHT           = "height";
IFrameSet_HORIZONTAL       = "h";
IFrameSet_VERTICAL         = "v";
IFrameSet_UP               = "u";
IFrameSet_DOWN             = "d";
IFrameSet_LEFT             = "l";
IFrameSet_RIGHT            = "r";
IFrameSet_GRIP_TITLE       = "Resize";
IFrameSet_PRE              = "iframeset_";
IFrameSet_PRE_AREA         = "area_";
IFrameSet_PRE_GRIP         = "grip_";
IFrameSet_AREA_TAG         = "div";
IFrameSet_AREA_WIDTH       = 20;
IFrameSet_AREA_HEIGHT      = 20;

/** The IFrame objects in order of appearance */
/*Array*/IFrameSet.prototype.iFrames = new Array();

/** The layout of the IFrameSet */
/*int*/IFrameSet.prototype.layout = IFrameSet_LAYOUT_1;

/** Defines whether the IFrame objects should be resizable */
/*boolean*/IFrameSet.prototype.resizable = true;

/** Defines whether user settings should be memorized using cookies */
/*boolean*/IFrameSet.prototype.allowCookies = true;

/** The server name cookies should be valid for */
/*String*/IFrameSet.prototype.cookieHost = IFrameSet_COOKIE_HOST;

/** The server path cookies should be valid for */
/*String*/IFrameSet.prototype.cookiePath = IFrameSet_COOKIE_PATH;

/** The time in days cookies should be valid for */
/*int*/IFrameSet.prototype.cookieTTL = IFrameSet_COOKIE_TTL;

/** The minimum width in pixels IFrame objects can be resized to */
/*int*/IFrameSet.prototype.minWidth = IFrameSet_MIN_SIZE;

/** The minimum height in pixels IFrame objects can be resized to */
/*int*/IFrameSet.prototype.minHeight = IFrameSet_MIN_SIZE;

/** The size in pixels to leave empty in the total window width */
/*int*/IFrameSet.prototype.offsetWidth = IFrameSet_OFFSET;

/** The size in pixels to leave empty in the total window height */
/*int*/IFrameSet.prototype.offsetHeight = IFrameSet_OFFSET;

/** The size in pixels of the resizing area between IFrame objects */
/*int*/IFrameSet.prototype.spacing = IFrameSet_SPACING;

/** The text displayed when the mouse is over the resizing area */
/*String*/IFrameSet.prototype.gripTitle = IFrameSet_GRIP_TITLE;

/** The icon displayed in the resizing area */
/*String*/IFrameSet.prototype.gripSrc = "";

/** The background color of the resizing area */
/*String*/IFrameSet.prototype.bgColor = "";

/**
 * Creates a new IFrameSet object with the specified ID.
 * @param id    The optional ID for the IFrameSet object
 * @return      The new IFrameSet object
 */
/*IFrameSet*/function IFrameSet(id)
	{
	if (!id)
		{
		id = IFrameSet_PRE+Util.createRandomId();
		this.allowCookies = false;
		}
	this.id            = id;
	this.resizeState   = IFrameSet_STATE_OFF;
	this.resizePosX    = IFrameSet_DEFAULT_POS;
	this.resizePosY    = IFrameSet_DEFAULT_POS;
	this.resizeAxis    = IFrameSet_VERTICAL;
	if (this.allowCookies)
            {
            var expMilli = new Date().valueOf()+(86400000*this.cookieTTL);
            var expDate = new Date(expMilli);
            this.cookieExp = expDate.toGMTString();
            }
	IFrameSet_set(this);
	//window.onresize = IFrameSet_refresh;
	document.onselectstart = IFrameSet_preventSelection; // fix for IE
	}

/*
 * Refreshes the IFrameSet
 */
/*void*/IFrameSet.prototype.refresh = function()
    {
    this.calc();
    for (var i in this.iFrames)
        {
        this.iFrames[i].refresh();
        }
    }

/*
 * Calculates the dimensions of the IFrame objects.
 */
/*void*/IFrameSet.prototype.calc = function()
    {
    if (this.allowCookies)
        {
        this.checkUserSettings();
        }
    var max = (this.layout != IFrameSet_LAYOUT_3) ? 2 : 3;
    switch (this.layout)
        {
        case IFrameSet_LAYOUT_1:
                if (!this.iFrames[0].fixedWidth)
                        {
                        if (this.iFrames[1].fixedWidth)
                                {
                                this.iFrames[0].width = Util.getWindowWidth()-this.iFrames[1].fixedWidth-3*this.spacing-2*this.iFrames[0].borderSize-2*this.iFrames[1].borderSize-this.offsetWidth;
                                }
                        else if (this.iFrames[1].percentWidth != IFrame_PERCENT)
                                {
                                this.iFrames[0].width = Util.getWindowWidth()-Math.round((Util.getWindowWidth()-3*this.spacing)*this.iFrames[1].percentWidth/100)-2*this.iFrames[0].borderSize-2*this.iFrames[1].borderSize-this.offsetWidth;
                                }
                        else
                                {
                                this.iFrames[0].width = Math.round((Util.getWindowWidth()-3*this.spacing)*this.iFrames[0].percentWidth/100)-2*this.iFrames[0].borderSize-this.offsetWidth;
                                }
                        }
                else
                        {
                        this.iFrames[0].width = this.iFrames[0].fixedWidth;
                        }
                if (!this.iFrames[0].fixedHeight)
                        {
                        this.iFrames[0].height = Util.getWindowHeight()-2*this.spacing-2*this.iFrames[0].borderSize-this.offsetHeight;
                        }
                else
	                {
	                this.iFrames[0].height = this.iFrames[0].fixedHeight;
	                }
                if (!this.iFrames[1].fixedWidth)
                        {
                        this.iFrames[1].width = Util.getWindowWidth()-this.iFrames[0].width-3*this.spacing-2*this.iFrames[1].borderSize-2*this.iFrames[0].borderSize-this.offsetWidth;
                        }
                else
	                {
	                this.iFrames[1].width = this.iFrames[1].fixedWidth;
	                }
                if (!this.iFrames[1].fixedHeight)
                        {
                        this.iFrames[1].height = Util.getWindowHeight()-2*this.spacing-2*this.iFrames[1].borderSize-this.offsetHeight;
                        }
                else
	                {
	                this.iFrames[1].height = this.iFrames[1].fixedHeight;
	                }
                break;
        case IFrameSet_LAYOUT_2:
            if (!this.iFrames[0].fixedWidth)
                {
                this.iFrames[0].width = Util.getWindowWidth()-2*this.spacing-2*this.iFrames[0].borderSize-this.offsetWidth;
                }
            else
                {
                this.iFrames[0].width = this.iFrames[0].fixedWidth;
                }
            if (!this.iFrames[0].fixedHeight)
                {
                if (this.iFrames[1].fixedHeight)
                    {
                    this.iFrames[0].height = Util.getWindowHeight()-this.iFrames[1].fixedHeight-3*this.spacing-2*this.iFrames[0].borderSize-2*this.iFrames[1].borderSize-this.offsetHeight;
                    }
                else if (this.iFrames[1].percentHeight != IFrame_PERCENT)
                    {
                    this.iFrames[0].height = Util.getWindowHeight()-Math.round((Util.getWindowHeight()-3*this.spacing)*this.iFrames[1].percentHeight/100)-2*this.iFrames[0].borderSize-2*this.iFrames[1].borderSize-this.offsetHeight;
                    }
                else
                    {
                    this.iFrames[0].height = Math.round((Util.getWindowHeight()-3*this.spacing-2*this.iFrames[0].borderSize-2*this.iFrames[1].borderSize)*this.iFrames[0].percentHeight/100)-this.offsetHeight;
                    }
                }
            else
                {
                this.iFrames[0].height = this.iFrames[0].fixedHeight;
                }
            if (!this.iFrames[1].fixedWidth)
                {
                this.iFrames[1].width = Util.getWindowWidth()-2*this.spacing-2*this.iFrames[1].borderSize-this.offsetWidth;
                }
            else
                {
                this.iFrames[1].width = this.iFrames[1].fixedWidth;
                }
            if (!this.iFrames[1].fixedHeight)
                {
                this.iFrames[1].height = Util.getWindowHeight()-this.iFrames[0].height-3*this.spacing-2*this.iFrames[1].borderSize-2*this.iFrames[0].borderSize-this.offsetHeight;
                }
            else
                {
                this.iFrames[1].height = this.iFrames[1].fixedHeight;
                }
            break;
        case IFrameSet_LAYOUT_3:
            if (!this.iFrames[0].fixedWidth)
                {
                if (this.iFrames[1].fixedWidth)
                    {
                    this.iFrames[0].width = Util.getWindowWidth()-this.iFrames[1].fixedWidth-3*this.spacing-2*this.iFrames[0].borderSize-2*this.iFrames[1].borderSize-this.offsetWidth;
                    }
                else if (this.iFrames[1].percentWidth != IFrame_PERCENT)
                    {
                    this.iFrames[0].width = Util.getWindowWidth()-Math.round((Util.getWindowWidth()-3*this.spacing)*this.iFrames[1].percentWidth/100)-2*this.iFrames[0].borderSize-2*this.iFrames[1].borderSize-this.offsetWidth;
                    }
                else
                    {
                    this.iFrames[0].width = Math.round((Util.getWindowWidth()-3*this.spacing)*this.iFrames[0].percentWidth/100)-2*this.iFrames[0].borderSize-this.offsetWidth;
                    }
                }
            else
                {
                this.iFrames[0].width = this.iFrames[0].fixedWidth;
                }
            if (!this.iFrames[0].fixedHeight)
                {
                this.iFrames[0].height = Util.getWindowHeight()-2*this.spacing-2*this.iFrames[0].borderSize-this.offsetHeight;
                }
            else
                {
                this.iFrames[0].height = this.iFrames[0].fixedHeight;
                }
            if (!this.iFrames[1].fixedWidth)
                {
                this.iFrames[1].width = Util.getWindowWidth()-this.iFrames[0].width-3*this.spacing-2*this.iFrames[1].borderSize-2*this.iFrames[0].borderSize-this.offsetWidth;
                }
            else
                {
                this.iFrames[1].width = this.iFrames[1].fixedWidth;
                }
            if (!this.iFrames[1].fixedHeight)
                {
                if (this.iFrames[2].fixedHeight)
                    {
                    this.iFrames[1].height = Util.getWindowHeight()-this.iFrames[2].fixedHeight-3*this.spacing-2*this.iFrames[1].borderSize-2*this.iFrames[2].borderSize-this.offsetHeight;
                    }
                else if (this.iFrames[2].percentHeight != IFrame_PERCENT)
                    {
                    this.iFrames[1].height = Util.getWindowHeight()-Math.round((Util.getWindowHeight()-3*this.spacing)*this.iFrames[2].percentHeight/100)-2*this.iFrames[1].borderSize-2*this.iFrames[2].borderSize-this.offsetHeight;
                    }
                else
                    {
                    this.iFrames[1].height = Math.round((Util.getWindowHeight()-3*this.spacing-2*this.iFrames[1].borderSize-2*this.iFrames[2].borderSize)*this.iFrames[1].percentHeight/100)-this.offsetHeight;
                    }
                }
            else
                {
                this.iFrames[1].height = this.iFrames[1].fixedHeight;
                }
            if (!this.iFrames[2].fixedWidth)
                {
                this.iFrames[2].width = this.iFrames[1].width;
                }
            else
                {
                this.iFrames[2].width = this.iFrames[2].fixedWidth;
                }
            if (!this.iFrames[2].fixedHeight)
                {
                this.iFrames[2].height = this.iFrames[0].height-this.iFrames[1].height-this.spacing-2*this.iFrames[2].borderSize;
                }
            else
                {
                this.iFrames[2].height = this.iFrames[2].fixedHeight;
                }
            break;
        }
    this.totalWidth = this.iFrames[0].width+(2*this.iFrames[0].borderSize);
    if (this.layout != IFrameSet_LAYOUT_2)
        {
        this.totalWidth += this.spacing+this.iFrames[1].width+(2*this.iFrames[1].borderSize);
        }
    this.totalHeight = this.iFrames[0].height+(2*this.iFrames[0].borderSize);
    if (this.layout == IFrameSet_LAYOUT_2)
        {
        this.totalHeight += this.spacing+this.iFrames[1].height+(2*this.iFrames[1].borderSize);
        }
    }

/**
 * Draws the IFrameSet object including its IFrame objects.
 */
/*void*/IFrameSet.prototype.draw = function()
	{
	var max = (this.layout != IFrameSet_LAYOUT_3) ? 2 : 3;
	var style = "margin:"+this.spacing+"px;";
	if (this.bgColor)
	    {
	    style += "background-color:"+this.bgColor+";";
	    }
	if (this.iFrames.length < max)
		{
		alert("IFrameSet: Not enough IFrame objects to draw.");
		}
	this.calc();
	document.write('<table border="0" cellspacing="0" cellpadding="0" style="'+style+'">');
	switch (this.layout)
		{
		case IFrameSet_LAYOUT_1:
			document.write('<tr>');
			for (var i=0; i < 2; i++)
				{
				document.write('<td valign="top">');
				this.iFrames[i].draw();
				document.write('</td>');
				if (i < 1)
					{
					this.drawGrip(IFrameSet_HORIZONTAL);
					}
				}
			document.write('</tr>');
			break;
		case IFrameSet_LAYOUT_2:
			for (var i=0; i < 2; i++)
				{
				document.write('<tr>');
				document.write('<td valign="top">');
				this.iFrames[i].draw();
				document.write('</td>');
				document.write('</tr>');
				if (i < 1)
					{
					document.write('<tr>');
					this.drawGrip(IFrameSet_VERTICAL);
					document.write('</tr>');
					}
				}
			break;
		case IFrameSet_LAYOUT_3:
			document.write('<tr><td rowspan="3" valign="top">');
			this.iFrames[0].draw();
			document.write('</td>');
			this.drawGrip(IFrameSet_HORIZONTAL);
			document.write('<td valign="top">');
			this.iFrames[1].draw();
			document.write('</td></tr>');
			document.write('<tr>');
			this.drawGrip(IFrameSet_VERTICAL);
			document.write('</tr>');
			document.write('<tr><td valign="top">');
			this.iFrames[2].draw();
			document.write('</td></tr>');
			break;
		default:
			alert("IFrameSet: No valid layout found.");
		}
	document.write('</table>');	
	this.drawArea();
	}

/*
 * Draws the resizing area between the IFrame objects.
 * @param axis    The resizing axis
 */
/*void*/IFrameSet.prototype.drawGrip = function(axis)
	{
	var rs = (this.layout == IFrameSet_LAYOUT_3 && axis == IFrameSet_HORIZONTAL) ? ' rowspan="3" ' : ' ';
	var style = "cursor:"+this.getCursorStyle(axis)+";";
	var width  = (axis == IFrameSet_HORIZONTAL) ? this.spacing : "100%";
	var height = (axis == IFrameSet_VERTICAL)   ? this.spacing : "100%";
	document.write('<td'+rs+'');
	document.write('width="'+width+'" height="'+height+'" ');
	if (this.resizable)
		{
		document.write('ondragstart="IFrameSet_startResize(\''+this.id+'\',\''+axis+'\',event)" ');
		document.write('onmousedown="IFrameSet_startResize(\''+this.id+'\',\''+axis+'\',event)" ');
		document.write('onmouseup="IFrameSet_stopResize(\''+this.id+'\',event)" ');
		document.write('ondragstop="IFrameSet_stopResize(\''+this.id+'\',event)" ');
		document.write('id="'+IFrameSet_PRE_GRIP+this.id+axis+'" ');
		if (this.gripSrc)
			{
			document.write('><img width="'+width+'" height="'+height+'" border="0" style="'+style+'" title="'+this.gripTitle+'" src="'+this.gripSrc+'">');
			}
		else
			{
			document.write('title="'+this.gripTitle+'"');
			// hack to ensure spacing between iframes without an image
			style += (axis == IFrameSet_HORIZONTAL) ? "font-size:1px;line-height:1px;" : "font-size:"+this.spacing+"px;line-height:"+this.spacing+"px;";
			document.write('style="'+style+'">');
			if (axis == IFrameSet_HORIZONTAL)
				{
				document.write('<span style="visibility:hidden">');
				for (var i=0; i < this.spacing; i++)
					{
					document.write('_');
					}
				document.write('</span>');
				}
			// eoh
			}
		}
	document.write('</td>');
	}

/*
 * Draws a transparent layer as a helper for resizing.
 */
/*void*/IFrameSet.prototype.drawArea = function()
	{
	document.write('<div id="'+IFrameSet_PRE_AREA+this.id+'" ');
	document.write('onmouseover="IFrameSet_doResize(\''+this.id+'\',event)" ');
	document.write('onmousemove="IFrameSet_doResize(\''+this.id+'\',event)" ');
	document.write('onmouseup="IFrameSet_stopResize(\''+this.id+'\',event)" ');
	document.write('onclick="IFrameSet_stopResize(\''+this.id+'\',event)" ');
	document.write('style="position:absolute;visibility:visible;top:-'+IFrameSet_AREA_HEIGHT+'px;left:-'+IFrameSet_AREA_WIDTH+'px;width:'+IFrameSet_AREA_WIDTH+'px;height:'+IFrameSet_AREA_HEIGHT+'px;">');
	document.write('</div>');
	}

/*
 * Resizes the IFrame objects horizontally.
 * @param dir    The direction in which resizing should be performed
 * @param off    The number of pixels to resize
 */
/*void*/IFrameSet.prototype.resizeWestEast = function(dir,off)
	{
	if (typeof(off) != "undefined" && off > 0)
		{
		// resize horizontally based on arguments
		var left = Util.getElem(this.iFrames[0].id);
		var right = new Array( Util.getElem(this.iFrames[1].id));
		if (this.layout == IFrameSet_LAYOUT_3)
			{
			right[1] = Util.getElem(this.iFrames[2].id);
			}
		var leftWidth = Util.getWindowWidth(left);
		var rightWidth = Util.getWindowWidth(right[0]);
		leftWidth = parseInt(leftWidth);
		rightWidth = parseInt(rightWidth);
		if (dir == IFrameSet_LEFT)
			{
			var leftWidthNew = leftWidth-off;
			var rightWidthNew = rightWidth+off;
			if (leftWidthNew >= this.minWidth+this.spacing)
				{
				Util.setWidth(left,leftWidthNew);
				for (var i in right)
					{
					Util.setWidth(right[i],rightWidthNew);
					}
				}
			}
		else
			{
			var leftWidthNew = leftWidth+off;
			var rightWidthNew = rightWidth-off;
			if (leftWidthNew+this.minWidth+(2*this.spacing) < this.totalWidth)
				{
				for (var i in right)
					{
					Util.setWidth(right[i],rightWidthNew);
					}
				Util.setWidth(left,leftWidthNew);
				}
			}
		}
	}

/*
 * Resizes the IFrame objects vertically.
 * @param dir    The direction in which resizing should be performed
 * @param off    The number of pixels to resize
 */
/*void*/IFrameSet.prototype.resizeNorthSouth = function(dir,off)
	{
	if (typeof(off) != "undefined" && off > 0)
		{
		// resize vertically based on arguments
		var upper,lower;
		switch (this.layout)
			{
			case IFrameSet_LAYOUT_3:
				upper = Util.getElem(this.iFrames[1].id);
				lower = Util.getElem(this.iFrames[2].id);
				break;
			default:
				upper = Util.getElem(this.iFrames[0].id);
				lower = Util.getElem(this.iFrames[1].id);
			}
		var upperHeight = Util.getWindowHeight(upper);
		var lowerHeight = Util.getWindowHeight(lower);
		upperHeight = parseInt(upperHeight);
		lowerHeight = parseInt(lowerHeight);
		if (dir == IFrameSet_UP)
			{
			var upperHeightNew = upperHeight-off;
			var lowerHeightNew = lowerHeight+off;
			if (upperHeightNew >= this.minHeight+this.spacing)
				{
				Util.setHeight(upper,upperHeightNew);
				Util.setHeight(lower,lowerHeightNew);
				}
			}
		else
			{
			var upperHeightNew = upperHeight+off;
			var lowerHeightNew = lowerHeight-off;
			if (upperHeightNew+this.minHeight+(2*this.spacing) < this.totalHeight)
				{
				Util.setHeight(lower,lowerHeightNew);
				Util.setHeight(upper,upperHeightNew);
				}
			}
		}
	}

/*
 * Shows the resizing helper layer.
 */
/*void*/IFrameSet.prototype.showArea = function()
	{
	var area = Util.getElem(IFrameSet_PRE_AREA+this.id)
	var left = Util.getElem(this.iFrames[0].id);
	Util.setProp(area,"cursor",this.getCursorStyle());
	Util.setProp(area,"width",this.totalWidth);
	Util.setProp(area,"height",this.totalHeight);
	Util.setProp(area,"top",Util.getPosY(left));
	Util.setProp(area,"left",Util.getPosX(left));
	Util.setProp(area,"visibility","visible");
	}

/*
 * Hides the resizing helper layer.
 */
/*void*/IFrameSet.prototype.hideArea = function()
	{
	var area = Util.getElem(IFrameSet_PRE_AREA+this.id);
	Util.setProp(area,"visibility","hidden");
	}

/*
 * Checks whether the IFrameSet is currently being resized.
 * @return    <code>true</code> if the IFrameSet object is being resized,
 *            or <code>false</code> otherwise
 */
/*boolean*/IFrameSet.prototype.isResized = function()
	{
	return (this.resizeState == IFrameSet_STATE_ON);
	}

/*
 * Selects the cursor style depending on the resizing axis.
 * @param axis    The resizing axis
 * @return        The cursor style
 */
/*String*/IFrameSet.prototype.getCursorStyle = function(axis)
	{
	if (!axis) axis = this.resizeAxis;
	return (axis == IFrameSet_VERTICAL) ? "s-resize" : "e-resize";
	}

/*
 * Checks for saved user settings and resizes IFrame objects accordingly.
 */
/*void*/IFrameSet.prototype.checkUserSettings = function()
	{
	for (var i in this.iFrames)
		{
		var id = this.iFrames[i].id;
		var width = Util.getCookie(this.id+"_"+id+"_"+IFrameSet_WIDTH);
		if (width && width.indexOf(":") > 0)
			{
			var tmp = width.split(":");
			var winWidth = parseInt(tmp[1]);
			if (winWidth >= Util.getWindowWidth()-2 && winWidth <= Util.getWindowWidth()+2)
				{
				width = parseInt(tmp[0]);
				if (width) this.iFrames[i].fixedWidth = width; 
				}
			else
				{
				// window width has changed -> delete setting
				this.deleteUserSetting(this.id+"_"+id+"_"+IFrameSet_WIDTH);
				}
			}
		var height = Util.getCookie(this.id+"_"+id+"_"+IFrameSet_HEIGHT);
		if (height && height.indexOf(":") > 0)
			{
			var tmp = height.split(":");
			var winHeight = parseInt(tmp[1]);
			if (winHeight >= Util.getWindowHeight()-2 && winHeight <= Util.getWindowHeight()+2)
				{
				height = parseInt(tmp[0]);
				if (height) this.iFrames[i].fixedHeight = height;
				}
			else
				{
				// window height has changed -> delete setting
				this.deleteUserSetting(this.id+"_"+id+"_"+IFrameSet_HEIGHT);
				}
			}
		}
	}

/*
 * Saves the user settings.
 */
/*void*/IFrameSet.prototype.saveUserSettings = function()
	{
	var winWidth = Util.getWindowWidth();
	var winHeight = Util.getWindowHeight();
	for (var i in this.iFrames)
		{
		var iFrame = Util.getElem(this.iFrames[i].id);
		this.saveUserSetting(this.id+"_"+iFrame.id+"_"+IFrameSet_WIDTH,parseInt(Util.getProp(iFrame,IFrameSet_WIDTH))+":"+winWidth);
		this.saveUserSetting(this.id+"_"+iFrame.id+"_"+IFrameSet_HEIGHT,parseInt(Util.getProp(iFrame,IFrameSet_HEIGHT))+":"+winHeight);
		}
	}

/*
 * Saves a specific user setting.
 * @param name    The name of the setting
 * @param value   The value of the setting
 */
/*void*/IFrameSet.prototype.saveUserSetting = function(name,value)
	{
	Util.setCookie(name,value,this.cookieHost,this.cookiePath,this.cookieExp);
	}

/*
 * Deletes a specific user setting.
 * @param name    The name of the setting
 */
/*void*/IFrameSet.prototype.deleteUserSetting = function(name)
	{
	Util.setCookie(name,"",this.cookieHost,this.cookiePath,new Date("November 26, 1974").toGMTString());
	}

//-----------------------------------------------------------------------------


/**
 * IFrame Documentation
 *
 * The IFrame object basically works like the HTML element <iframe>. It can
 * be added to an IFrameSet object to achieve what the <frameset> element
 * can do with <frame> elements: Draw them in a specified layout.
 *
 * Parameters:
 *
 * - id            The ID for the IFrame object. It must be unique throughout
 *                 the document and may be alpha-numeric, as long as the first
 *                 character is a letter. The ID is case-sensitive. If left
 *                 empty or undefined, it is generated using the following
 *                 form: iframe_[n] (where [n] is an auto-incrementing, zero-
 *                 based digit).
 *                 Note:    You will only be able to reference IFrame objects
 *                          with specific IDs.
 *
 * Required properties:
 *
 * - src           The source to the document shown in the IFrame.
 *                 Default value is an empty String.
 *
 * Optional properties:
 *
 * - fixedWidth    The width in pixels the IFrame object should use. Default
 *                 value is 0, which the IFrameSet object will render at its
 *                 default percentage width.
 *
 * - fixedHeight   The height in pixels the IFrame object should use. Default
 *                 value is 0, which lets the IFrameSet object calculate the
 *                 height based of the size of the window and the other IFrame
 *                 objects.
 *
 * - percentWidth  The window width in percent the IFrame object should use if
 *                 the window is split horizontally. Default value is 50.
 *
 * - percentHeight The window height in percent the IFrame object should use if.
 *                 the window is split vertically. Default value is 50.
 *
 * - scrolling     The scrolling behavior if the IFrame object.
 *                 Accepted values are:
 *                     IFrame_SCROLLING_AUTO  Scrolling will be enabled
 *                                            depending on IFrame dimensions
 *                                            (Default)
 *                     IFrame_SCROLLING_YES   Scrolling will always be enabled
 *                     IFrame_SCROLLING_NO    Scrolling will always be disabled
 *
 * - className     The name of the CSS class the IFrame object should use.
 *                 Default value is an empty String.
 *
 * - borderSize    The border size in pixels the IFrame object is surrounded
 *                 by. This property should only be used if the CSS class 
 *                 assigned to the IFrame object defines a border. Default 
 *                 value is 0.
 *
 * Methods:
 *
 * - draw()        Draws the IFrame object to the HTML document.
 */

// constants -------------------------------------------------------------------
IFrame_WIDTH = 0;
IFrame_HEIGHT = 0;
IFrame_PERCENT = 50;
IFrame_PRE = "iframe_";
IFrame_SCROLLING_AUTO = "auto";
IFrame_SCROLLING_YES = "yes";
IFrame_SCROLLING_NO = "no";

/** The source document of the IFrame object */
/*String*/IFrame.prototype.src = "";

/** The effective width of the IFrame object in pixels. Do not set! */
/*int*/IFrame.prototype.width = IFrame_WIDTH;

/** The effective height of the IFrame object in pixels. Do not set! */
/*int*/IFrame.prototype.height = IFrame_HEIGHT;

/** The fixed width of the IFrame object in pixels */
/*int*/IFrame.prototype.fixedWidth = IFrame_WIDTH;

/** The fixed height of the IFrame object in pixels */
/*int*/IFrame.prototype.fixedHeight = IFrame_HEIGHT;

/** The window width of the IFrame object in percent */
/*int*/IFrame.prototype.percentWidth = IFrame_PERCENT;

/** The window height of the IFrame object in percent */
/*int*/IFrame.prototype.percentHeight = IFrame_PERCENT;

/** The scrolling behavior of the IFrame object */
/*String*/IFrame.prototype.scrolling = IFrame_SCROLLING_AUTO;

/** The name of the CSS class the IFrame object should use */
/*String*/IFrame.prototype.className = "";

/** The size of the border defined in the specified class in pixels */
/*int*/IFrame.prototype.borderSize = 0;

/**
 * Creates a new IFrame object with the specific ID.
 * @param id    The optional ID for the IFrame object
 * @return      The new IFrame object
 */
/*IFrame*/function IFrame(id)
	{
	if (!id)
		{
		id = IFrame_PRE+Util.createRandomId();
		}
	this.id = id;
	}

/**
 * Refreshes the IFrame object.
 */
/*void*/IFrame.prototype.refresh = function()
	{
	var iFrame = document.getElementById(this.id);
	Util.setProp(iFrame, "width", this.width);
	Util.setProp(iFrame, "height", this.height);
	}


/**
 * Draws the IFrame object.
 */
/*void*/IFrame.prototype.draw = function()
	{
	document.write('<iframe src="'+this.src+'" ');
	document.write('id="'+this.id+'" name="'+this.id+'" ');
	if (this.className)
		{
		document.write('class="'+this.className+'" ');
		}
	document.write('frameborder="no" ');
	document.write('scrolling="'+this.scrolling+'" ');
	document.write('style="width:'+this.width+';height:'+this.height+'">');
	document.write('</iframe>');
	}

//-----------------------------------------------------------------------------
