/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
/**
 * NodeTree constructor.
 * @param name     the name of the node tree
 * @param provider the html providerclass
 * @param levelURI the location to fetch level data
 */
function NodeTree(/*string*/ name, /*string*/ provider, /*string*/ levelURI) {
    this.name = name;
    this.provider = provider;
    this.levelURI = levelURI;
    this.selectedNode = null;
    this.onSelectNode = null;
}

/**
 * Fetches a page from a url
 * @param url the location
 *
 * @return the fetched page
 */
/*string*/ NodeTree.prototype.httpGet = function(/*string*/ url) {
    var httpcon = document.all ? new ActiveXObject("Microsoft.XMLHTTP") : new XMLHttpRequest();
    if (httpcon) {
	httpcon.open('GET', url, false);
	httpcon.send(null);
	return httpcon.responseText;
    } else {
	return "";
    }
};

/**
 * Opens a node
 * @param node the node to open
 */
NodeTree.prototype.openNode = function(/*DHTMLObject*/ node, /*boolean*/ callOnSelect, /*boolean*/ immediate) {
	// if leaf, don't do anything
        if (!node || node.className.indexOf("Leaf")>0) {
	    // execute callback, if defined
	    if (callOnSelect && this.onSelectNode) {
		this.onSelectNode(this.name, this.selectedNode);
	    }
	    return;
	}

	var subnodes = NodeTree.getSubNodes(node);

	// open node
	node.className = node.className.replace("Closed", "Open");

	// open subnode
	subnodes.className = subnodes.className.replace("Closed", "Open");

	// load html of sub node
	if (subnodes.innerHTML=="") {
	    subnodes.innerHTML="<font color=\"888888\">loading...</font>";
	    var path=encodeURIComponent(this.getNodePath(node));
        if (immediate) {
	        this.loadSubNodes(path, callOnSelect)
        } else {
            setTimeout("NodeTree.getTree(\""+this.name+"\").loadSubNodes(\""+path+"\", "+callOnSelect+");", 1);
        }
	} else {
	    // execute callback, if defined
	    if (callOnSelect && this.onSelectNode) {
		this.onSelectNode(this.name, this.selectedNode);
	    }
	}
};
/**
 * Loads sub nodes from the server and fills the subnodes DIV
 * @param path the path of the node
 */
NodeTree.prototype.loadSubNodes = function(/*string*/ path, /*boolean*/ callOnSelect) {
    path = decodeURIComponent(path);
    var url = this.levelURI;
    url+="?Path=" + encodeURIComponent(path);
    url+="&TreeName=" + encodeURIComponent(this.name);
    url+="&Provider=" + encodeURIComponent(this.provider);
    url+="&_charset_=utf8";
    url+="&ck="+(new Date()).valueOf();

    var subnodes = this.getSubNodesByPath(path);
    subnodes.innerHTML = null;
    subnodes.innerHTML = this.httpGet(url);

    // execute callback, if defined
    if (callOnSelect && this.onSelectNode) {
	this.onSelectNode(this.name, this.selectedNode);
    }
};

/**
 * Reloads the entire tree
 */
NodeTree.prototype.reload = function() {
    // get root node
    var root = document.getElementById("NodeTree_"+this.name).firstChild;
    this.reloadNode(root, this.selectedNode);
};

/**
 * Reloads a node
 * @param path the path of the node
 */
NodeTree.prototype.reloadNodeByPath = function(/*string*/ path) {
    this.reloadNode(this.getNodeByPath(path));
};

/**
 * Selects a node by path
 * @param path the path of the node
 */
NodeTree.prototype.selectNodeByPath = function(/*string*/ path) {
    var i=1;
    while (i>0) {
    	var subpath = path.substring(0, i);
    	var node = this.getNodeByPath(subpath);
    	this.openNode(node, false, true);
	i=path.indexOf('/', i+1);
    }
    var node = this.getNodeByPath(path);
    if (node && node.firstChild) {
    	this.selectNode(node.firstChild, true);
    }
};

/**
 * Reloads a node
 * @param node the node to reload
 * @param openNode the path of the node that is open
 */
NodeTree.prototype.reloadNode = function(/*DHTMLObject*/ node, /*string*/ openNode) {
    if (!node) return;

    var url = this.levelURI;
    var path = this.getNodePath(node);
    url+="?Path=" + encodeURIComponent(path);
    if (!openNode && node.className.indexOf("Open")>=0) {
    	openNode=path;
    }
    url+="&_charset_=utf8";
    url+="&OpenNode=" + encodeURIComponent(openNode);
    if (path==this.selectedNode) {
    	url+="&SelectedNode=" + encodeURIComponent(path);
    }
    url+="&NodeOnly=1";
    url+="&Provider=" + encodeURIComponent(this.provider);
    url+="&TreeName=" + encodeURIComponent(this.name);
    url+="&ck="+(new Date()).valueOf();

    var subnodes = NodeTree.getSubNodes(node);

    node.outerHTML = this.httpGet(url);
    this.loadSubNodes(path);
};

/**
 * Closes a node
 * @param node the node to close
 */
NodeTree.prototype.closeNode = function(/*DHTMLObject */ node) {
    // close subnodes
    var subnodes = NodeTree.getSubNodes(node);
    subnodes.className = subnodes.className.replace("Open", "Closed");

    // close node
    node.className = node.className.replace("Open", "Closed");
};

/**
 * Open or Closes a node
 * @param node the node to open or close
 * @param evt the event that caused the toggle
 */
NodeTree.prototype.toggleNode = function(/*DHTMLObject*/ node, /*DHTMLEvent*/ evt) {
    if (window.event) {
	window.event.cancelBubble=true;
    } else {
	evt.stopPropagation();
    }

    // ignore clicks 'below' node
    if (typeof(evt.offsetY)=="undefined" || evt.offsetY<=16) {
	if (node.className.indexOf("nOpen")>=0) {
	    this.closeNode(node);
	} else {
	    this.openNode(node);
	}
    }
};

/**
 * selects a node
 * @param nodeItem the dhtml element that was selected
 * @param openNode indicates if the node has to be opened
 * @param evt the dhtml event
 */
NodeTree.prototype.selectNode = function(/*DHTMLObject*/ nodeItem,
    /*boolean*/ openNode, /*DHTMLEvent*/ evt) {
    if (window.event) {
	window.event.cancelBubble=true;
    } else if (evt) {
	evt.stopPropagation();
    }

    // get nodename element
    var nodeName = nodeItem.firstChild.nextSibling;
    if (nodeName.className.indexOf(" nnSelected")<0) {
    	// unselect last name
    	if (this.selectedNode) {
	    var selNode = this.getNodeByPath(this.selectedNode);
	    if (selNode) {
	    	var selNodeName = selNode.firstChild.firstChild.nextSibling;
	    	selNodeName.className = selNodeName.className.replace(" nnSelected", "");
	    }
    	}
    	nodeName.className+=" nnSelected";

	// get the corresponding node  */
	var node = nodeItem.parentNode;

	// remeber selection
	this.selectedNode = this.getNodePath(node);

        if (openNode) {
            // open this node
            this.openNode(node, true);
        }
        else {
            // execute callback, if defined
            var callOnSelect = true;
            if (this.onSelectNode) {
                this.onSelectNode(this.name, this.selectedNode);
            }
        }
    }
};

/**
 * registers the function that will be called if a node gets selected.
 * its signature is: function(String treeName, String path)
 * @param callback the callnack to register
 */
NodeTree.prototype.registerOnSelectNode = function(/*function*/ callback) {
    this.onSelectNode = callback;
};

/**
 * sets the selected node
 * @param path the path of the selected node
 */
NodeTree.prototype.setSelectedNode = function(/*string*/ path) {
    this.selectedNode = path;
};

/**
 * gets the node element of the given path
 * @param path the path of the node
 */
/*DHTMLObject*/ NodeTree.prototype.getNodeByPath = function(/*string*/ path) {
    var id = this.name + ":" + NodeTree.escape(path);
    return document.getElementById(id);
};

/**
 * gets the path of the given node element
 * @param node the node element
 */
/*string*/ NodeTree.prototype.getNodePath = function(/*DHTMLObject*/ node) {
    var pos=node.id.indexOf(":");
    return NodeTree.unescape(pos>=0 ? node.id.substring(pos+1) : node.id);
};

/**
 * gets the subnodes element of the given node path
 * @param path the path of the node
 */
/*DHTMLObject*/ NodeTree.prototype.getSubNodesByPath = function(/*string*/ path) {
    var node = this.getNodeByPath(path);
    return node.nextSibling;
};


//-----------------------------------------------------------------------------
// static members

/** map containing all the registred tree controls */
NodeTree.controls = new Object();


//-----------------------------------------------------------------------------
// static functions

/*string*/ NodeTree.escape = function(/*string*/ str) {
    return str;
    str = encodeURIComponent(str);
    str = str.replace("_", "%5f");
    str = str.replace("%", "_");
    return str;
};

/*string*/ NodeTree.unescape = function(/*string*/ str) {
    return str;
    str = str.replace("_", "%");
    str = decodeURIComponent(str);
    return str;
};

/**
 * Creates a new NodeTree and puts it into the control registry
 * @param name     name of new node tree
 * @param provider the html provider
 * @param levelURI the location to fetch level data
 *
 * @return newly created node tree instance
 */
/*NodeTree*/ NodeTree.createTree = function(/*string*/ name, /*string*/ provider, /*string*/ levelURI) {
	return this.controls[name] = new NodeTree(name, provider, levelURI);
};

/**
 * Returns the tree control with the given name
 * @param name the name of the control to return
 *
 * @return the tree control with the given name
 */
/*NodeTree*/ NodeTree.getTree = function(/*String*/ name) {
	return this.controls[name];
};

/**
 * Returns the tree controls object for a given node.
 * @param node the node
 * @return the tree control object for a given node.
 */
/*NodeTree*/ NodeTree.getTreeByNode = function(/*DHTMLObject*/ node) {
	// walk up the DHTML tree until the root div is found.
	while (node.className.indexOf("nodeTree" < 0)) {
		node = node.parentNode;
	}

	// get name of control
	var name = node.id.substring(node.id.indexOf(":")+1);
	return this.getTree(name);
};

/**
 * gets the subnodes element of the given node
 * @param node the node
 */
/*DHTMLObject*/ NodeTree.getSubNodes = function(/*DHTMLObject*/ node) {
    return node.nextSibling;
};

/**
 * Open or Closes a node
 * @param name the name of the tree
 * @param node the node to open or close
 * @param evt the event that caused the toggle
 */
NodeTree.toggleNode = function(/*string*/name, /*DHTMLObject*/ node, /*DHTMLEvent*/ evt) {
    this.getTree(name).toggleNode(node, evt);
};

/**
 * selects a node
 * @param name the name of the tree
 * @param elem the dhtml element that was selected
 * @param openNode indicates if the node has to be opened
 * @param evt the dhtml event
 */
NodeTree.selectNode = function(/*string*/ name, /*DHTMLObject*/ elem,
    /*boolean*/ openNode, /*DHTMLEvent*/ evt) {
    this.getTree(name).selectNode(elem, openNode, evt);
};

/**
 * adds a CSS class to the DHTMLObject
 * @param elem the DHTML object
 * @param cssClass the className
 */
NodeTree.addClass = function(/*DHTMLObject*/ elem, /*string*/ cssClass) {
    if (elem.className.indexOf(cssClass)<0) {
    	elem.className+= " " + cssClass;
    }
};

/**
 * removes a CSS class from the DHTMLObject
 * @param elem the DHTML object
 * @param cssClass the className
 */
NodeTree.removeClass = function(/*DHTMLObject*/ elem, /*string*/ cssClass) {
    if (elem.className.indexOf(" " + cssClass)>=0) {
    	elem.className = elem.className.replace(" " + cssClass, "");
    } else if (elem.className.indexOf(cssClass)>=0) {
    	elem.className = elem.className.replace(cssClass, "");
    }
};
