/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
/**
 * Implements the package share client
 * @class PackageShareClient
 *
 * @constructor
 */
var PackageShareClient = function(ctxPath, sharePath) {
    this.sharePath = sharePath;
    this.ctxPath = ctxPath;
};

PackageShareClient.STATUS_MISSING = "missing";

PackageShareClient.STATUS_OUTDATED = "outdated";

PackageShareClient.STATUS_NOT_NEWEST = "notNewest";

PackageShareClient.STATUS_DOWNLOADING = "downloading";

PackageShareClient.STATUS_ADDING = "adding";

PackageShareClient.STATUS_DOWNLOADED = "downloaded";

PackageShareClient.STATUS_INSTALLED = "installed";

PackageShareClient.STATUS_UPLOADING = "uploading";

PackageShareClient.STATUS_UPLOADED = "uploaded";

PackageShareClient.STATUS_UPLOAD_ERROR = "uploadError";

PackageShareClient.prototype = {

    // map of all packages, key = pid
    packsByPid: {},

    // map of all local packages, key = pid
    localPacks: {},

    // map of newest server packages, key = "group:name"
    serverPacks: {},

    sharePath: "",

    ctxPath: "",

    xmlhttp: null,

    /**
     * Invoked by package share when a package button is clicked
     * @param pid package id
     */
    onPackageClick: function(pid) {
        var pack = this.packsByPid[pid];
        if (!pack) {
            return;
        }
        switch (pack.status) {
            case PackageShareClient.STATUS_MISSING:
                this.download(pack);
                break;
            case PackageShareClient.STATUS_OUTDATED:
                this.update(pack);
                break;
            case PackageShareClient.STATUS_NOT_NEWEST:
                this.downgrade(pack);
                break;
            case PackageShareClient.STATUS_DOWNLOADING:
                // do nothing
                break;
            case PackageShareClient.STATUS_ADDING:
                // do nothing
                break;
            case PackageShareClient.STATUS_DOWNLOADED:
            case PackageShareClient.STATUS_INSTALLED:
                this.open(pack);
                break;
        }
    },

    /**
     * Invoked by package share to navigate to package manager home
     */
    onHome: function() {
        window.location.href = this.ctxPath + "/packmgr/index.jsp";
    },

    /**
     * Invoked by package share to register a package on the current list
     * @param pid package id
     * @param group package group
     * @param name package name
     * @param vers package version
     * @param url package download url
     * @param size package size
     */
    onPackageRegister: function(sPack) {
        var pid = sPack.pid;
        var key = sPack.group + ":" + sPack.name;

        // Check if package exists locally
        var pack = this.packsByPid[pid];
        var existsLocally = false;
        if (!pack) {
            // Package does not exist locally
            pack = {
                sPack: sPack,
                pid: sPack.pid,
                group: sPack.group,
                name: sPack.name,
                sv: sPack.version,
                sb: sPack.build,
                url: sPack.path,
                size: sPack.size,
                status: PackageShareClient.STATUS_MISSING
            };
        } else {
            // Package exists locally
            existsLocally = true;
            pack.sPack = sPack;
            pack.sv = sPack.version;
            pack.sb = sPack.build;
            pack.size = sPack.size;
            pack.url = sPack.path;
            if (pack.sv > pack.lv) {
                pack.status = PackageShareClient.STATUS_OUTDATED;
            } else if (pack.sv == pack.lv) {
                if (pack.lb && pack.sb && pack.lb < pack.sb) {
                    pack.status = PackageShareClient.STATUS_OUTDATED;
                }
            }
        }
        this.packsByPid[pid] = pack;

        // Check if current server package is the latest one (from version and build number)
        var csPack = this.serverPacks[key];
        if (csPack) {
            if (sPack.version > csPack.version) {
                this.serverPacks[key] = sPack;
            } else if (sPack.version == csPack.version) {
                if (sPack.build && csPack.build) {
                    if (sPack.build > csPack.build) {
                        this.serverPacks[key] = sPack;
                    } else if (sPack.build < csPack.build) {
                        this.updateConcurrentPackageStatus(csPack, pack, existsLocally);
                    }
                }
            } else if (sPack.version < csPack.version) {
                this.updateConcurrentPackageStatus(csPack, pack, existsLocally);
            }
        } else {
            this.serverPacks[key] = sPack;
        }

        this.updateStatus(pack);
    },

    /**
     * Adds a local package. assumes that the package is not registered
     * @param pid the package id
     * @param group group name
     * @param name package name
     * @param vers version
     */
    onLocalPackage: function(pid, group, name, vers, installed, size, sizeStr, build) {
        this.packsByPid[pid] = this.localPacks[pid] = {
            pid: pid,
            group: group,
            name: name,
            lv: vers,
            lb: build,
            url: "",
            size: size,
            sizeStr: sizeStr,
            status: installed
                    ? PackageShareClient.STATUS_INSTALLED
                    : PackageShareClient.STATUS_DOWNLOADED
        };
    },

    /**
     * Invoked by package share to log out the current package share session
     */
    onLogout: function() {
        window.location.href = this.ctxPath + this.sharePath + "/logout.html";
    },

    /**
     * Returns the list of local packages
     */
    getLocalPackages: function() {
        var local = {};
        for (var pid in this.packsByPid) {
            var p = this.packsByPid[pid];
            if (p.status != PackageShareClient.STATUS_MISSING) {
                local[pid] = p;
            }
        }
        return local;
    },

    /**
     * invoked by package share client to get the upload url
     */
    getUploadURL: function() {
        return this.ctxPath + "/packageshare/upload";
    },

    /**
     * Invoked by package share client to trigger the transfer of a previously
     * uploaded package.
     * @param sPack the (dummy) server package
     */
    onUpload: function(sPack) {
        this.packsByPid[sPack.pid] = {
            sPack: sPack,
            pid: sPack.pid,
            size: sPack.size,
            group: sPack.group
        };
        var url = this.ctxPath + "/packmgr/service/console.html";
        this.openConsole(url, {
            cmd: "shareUpload",
            path: sPack.path,
            group: sPack.group,
            name: sPack.name,
            pid: sPack.pid
        });
    },

    /**
     * little hack to inform the package browser that upload was finished
     */
    doUploadFinished: function() {
    },

    updateStatus: function(pack) {
        pack.sPack.draw(pack.sPack, pack.status, pack.progress, pack.progressMax, pack.msg);
    },

    updateConcurrentPackageStatus: function(csPack, pack, existsLocally) {
        if (!this.localPacks[csPack.pid]) {
            // Latest server package does not exist locally
            if (existsLocally) {
                // Current package exists locally
                var outDatedPack = this.packsByPid[csPack.pid];
                outDatedPack.status = PackageShareClient.STATUS_OUTDATED;
                this.updateStatus(outDatedPack);
            }
        } else {
            // Latest server package does not exist locally
            if (!existsLocally) {
                // Current package does not exist locally
                pack.status = PackageShareClient.STATUS_NOT_NEWEST;
            }
        }
    },

    /**
     * Prompts if ok to download, then starts the download
     * @param pack the package
     */
    download: function(pack) {
        if (this.validateDownload(pack) && confirm("OK to download package: " + pack.pid + " ?")) {
            this.doDownload(pack);
        }
    },

    /**
     * Prompts if ok to update, then starts the download
     * @param pack the package
     */
    update: function(pack) {
        if (this.validateDownload(pack) && confirm("OK to update package: " + pack.pid + " ?")) {
            this.doDownload(pack);
        }
    },

    /**
     * Prompts if ok to downgrade, then starts the download
     * @param pack the package
     */
    downgrade: function(pack) {
        if (this.validateDownload(pack) && confirm("Your local version seems to be newer than this one. OK to downgrade package: " + pack.pid + " ?")) {
            this.doDownload(pack);
        }
    },

    /**
     * Validates if the given package can be downloaded, i.e. installed in the system.
     * @param pack the package
     * @return false if not ok
     */
    validateDownload: function(pack) {
        var xmlhttp = this.getXmlHttp();
        if (!xmlhttp) {
            return false;
        }

        if (xmlhttp.readyState < 4) {
            xmlhttp.abort();
        }

        // send the authentication request
        var params = "cmd=validateDownload";
        params += "&pid=" + encodeURIComponent(pack.pid);
        xmlhttp.open('POST', this.ctxPath + "/packmgr/service/console.json", false);
        xmlhttp.setRequestHeader("Content-type", "application/x-www-form-urlencoded");
        xmlhttp.send(params);
        if (xmlhttp.status != 200) {
            alert("Error while validating download. Check logs for more information.");
            return false;
        }
        var response = eval("(" + xmlhttp.responseText + ")");
        if (response.success) {
            return true;
        } else {
            alert(response.msg);
            return false;
        }
    },

    /**
     * opens the download "console"
     * @param pack the package
     */
    doDownload: function(pack) {
        var url = this.ctxPath + "/packmgr/service/console.html";
        pack.cmd="shareDownload";
        pack.pid = pack.sPack.pid;
        this.openConsole(url, pack);
    },

    /**
     * Opens the respective package page
     * @param pack the package
     */
    open: function(pack) {
        window.location.href = this.ctxPath + "/packmgr/index.jsp#" + pack.pid;
    },

    /**
     * Invoked by a console to signal a status change
     * @param pid the package id
     * @param stat the (new) status
     * @param progress optional progress indicator
     */
    onStatus: function(pid, stat, progress, max, msg) {
        var pack = this.packsByPid[pid];
        if (!pack) {
            return;
        }
        pack.status = stat;
        pack.progress = progress;
        pack.progressMax = pack.size;
        pack.msg = msg;
        this.updateStatus(pack);
    },

    openConsole: function(url, ps) {
        var frm = document.getElementById("console_form");
        frm.action = url;
        for (var name in ps) {
            var field = frm[name];
            if (field) {
                field.value = ps[name];
            }
        }
        frm.submit();
    },

    onKeyDown:  function(e) {
        if (window.event) e = window.event; // msie
        if (e.keyCode == "C".charCodeAt(0) && e.ctrlKey && e.shiftKey) {
            PackageShareClient.toggleConsole(true);
            e.preventDefault();
        }
    },

    getXmlHttp: function() {
        if (this.xmlhttp) {
            return this.xmlhttp;
        }

        if (window.XMLHttpRequest) {
            this.xmlhttp = new XMLHttpRequest();
        } else {
            if (window.ActiveXObject) {
                try {
                    this.xmlhttp = new ActiveXObject('Msxml2.XMLHTTP');
                } catch (ex) {
                    try {
                        this.xmlhttp = new ActiveXObject('Microsoft.XMLHTTP');
                    } catch (ex) {
                    }
                }
            }
        }
        return this.xmlhttp;
    }

};

PackageShareClient.toggleConsole = function(show) {
    //var btn = document.getElementById("cq-packagemgr-console-button");
    var csl = document.getElementById("cq-packagemgr-console");
    if (show) {
        //btn.className = "cq-packagemgr-console-button-header cq-packagemgr-console-hidden";
        csl.className = "cq-packagemgr-console";
    } else {
        //btn.className = "cq-packagemgr-console-button-header";
        csl.className = "cq-packagemgr-console cq-packagemgr-console-hidden";
    }
};