/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
/**
 * ToolBar
 *
 * Used to draw a toolbar with icons and/or text buttons.
 * Depends on JS classes in contextmenu.js.
 * Depends on CSS classes in toolbar.css unless custom classes are defined.
 */

/**
 * ToolBar constructor
 * @param name the name of the toolbar
 */
function ToolBar(name) {
    this.name = name;
    this.items = new Array();
    //this.contextMenu = ContextMenu.create(this.name+"_menu");
    //this.contextMenu.setOffset(document.all ? 0 : -15, 37);
    this.docroot = "";
    this.adaptToWindow = true;
}

/** CSS classes */
ToolBar.barClass = "toolbar"; // class for TABLE, TR and TD elements
ToolBar.buttonClass = "toolbarButton"; // class for A, SPAN and IMG elements
ToolBar.separatorClass = "toolbarSeparator"; // class for SPAN element
ToolBar.spacerClass = "toolbarSpacer"; // class for TD element
ToolBar.logoClass = "toolbarLogo"; // class for A and IMG elements
ToolBar.disabledClass = "toolbarDisabled"; // class for all elements

/**
 * Sets the docroot for the images and other resources
 * @param docroot the docroot to set
 */
ToolBar.prototype.setDocroot = function(/*String*/ docroot) {
    this.docroot = docroot;
}

/**
 * Adds a toolbar item to the bar
 * @param item the toolbar item to add
 */
ToolBar.prototype.addItem = function(/*ToolBarItem*/ item) {
    item.setDocroot(this.docroot);
    this.items.push(item);
    if (item.icon) {
        // preload icon
        var img = new Image();
        img.src = this.docroot+item.icon;
    }
}

/**
 * Returns the item of a given name or null if it doesn't exist
 * @param name the name of the item
 * @return the item
/*ToolBarItem*/ToolBar.prototype.getItem = function(/*String*/ name) {
    var item;
    for (var i in this.items) {
        if (this.items[i].name == name) {
            item = this.items[i];
            break;
        }
    }
    return item;
}

/**
 * Sets the 'enabled' state of the item specified by 'name'
 * @param name the name of the item
 * @param enable if true, enables item; if false, disables it
 */
/*void*/ToolBar.prototype.enableItem = function(/*String*/ name, /*boolean*/enable) {
    var item = this.getItem(name);
    if (item) {
        item.enable(enable);
    } else {
        // find multi button items and propagate enable action
        for (var i in this.items) {
            if (this.items[i] instanceof ToolBarMultiButton) {
                this.items[i].enableItem(name+"_MultiButtonItem",enable);
                this.items[i].populate(); // re-populate
            }
        }
    }
}

/**
 * Adds a logo to the right of the toolbar
 */
ToolBar.prototype.setLogo = function(/*ToolBarImage*/ logo) {
    this.logo = logo;
    this.logo.setDocroot(this.docroot);
}

/**
 * Draws the toolbar to the current 'document'
 */
ToolBar.prototype.draw = function() {
    document.write('<table class="'+ToolBar.barClass+'" cellspacing="0" cellpadding="0" width="100%"><tr valign="top"><td class="'+ToolBar.barClass+'">');
    document.write('<table border="0" cellspacing="0" cellpadding="0" height="36"><tr class="'+ToolBar.barClass+'" valign="top">');
    for (var i in this.items) {
        document.write(this.items[i].getOuterHTML());
    }
    if (this.adaptToWindow) {
        this.moreButton = new ToolBarMultiButton(this.name+"_more", "", "", "&raquo;",  "",  "Click for more options", true);
        this.moreButton.setDocroot(this.docroot);
        document.write(this.moreButton.getOuterHTML());
    }
    document.writeln('</tr></table></td>');
    //document.write('<td class="toolbarMore" align="right">');
    //document.writeln('</td>');
    if (this.logo)
        {
        document.writeln('<td class="'+ToolBar.barClass+'" width="1" align="right">'+this.logo.getInnerHTML()+'</td>');
        }
    document.writeln('</tr></table>');
    // populate all multi buttons
    for (var i in this.items) {
        if (this.items[i] instanceof ToolBarMultiButton) {
            this.items[i].populate();
        }
    }
    ToolBar.checkToolBarSpace(this);
}

/** static container of all toolbars */
ToolBar.toolbars = new Object();

/**
 * creates a new toolbar with the given name
 * @param name the name for the new toolbar
 * @return the newly created toolbar
 */
/*ToolBar*/ ToolBar.createToolBar = function(name) {
    return this.toolbars[name] = new ToolBar(name);
}

/**
 * gets the toobar from the global container
 * @param name the name of the toolbar to retrieve
 * @return the indicated toolbar
 */
/*ToolBar*/ ToolBar.getToolBar = function(name) {
    return this.toolbars[name];
}

/**
 * checks all toolbars in global container if they
 * fit the current window width and hides items on
 * the right until they do.
 * @see ToolBar#checkToolBarSpace(ToolBar)
 */
/*void*/ ToolBar.refresh = function() {
    ContextMenu.hideAll();
    for (var i in ToolBar.toolbars) {
        ToolBar.checkToolBarSpace(ToolBar.toolbars[i]);
    }
}

/**
 * checks a specific toolbar if it fits the current
 * window width and hides items on the right until
 * it does.
 * @param toolbar   the toolbar to check
 */
/*void*/ ToolBar.checkToolBarSpace = function(/*ToolBar*/toolbar) {
    if (!toolbar || !(toolbar instanceof ToolBar)
            || !toolbar.adaptToWindow) return;
    var more = Util.getElem(toolbar.moreButton.name);
    Util.setProp(more,"display","none"); // hide more button
    toolbar.moreButton.clear();
    for (var i in toolbar.items) { // recheck all from scratch
        if (!toolbar.items[i].isDisplayed) {
            var elem = Util.getElem(toolbar.items[i].name);
            Util.setProp(elem,"display","inline");
            toolbar.items[i].isDisplayed = true;
        }
    }
    var tolerance = (toolbar.logo) ? 107 : 0;
    var logoX = 0;
    logoX = Util.getPosX(Util.getElem(toolbar.logo.name));
    if (logoX+tolerance > Util.getWindowWidth()) {
        tolerance += 32; // add more button
        var i = toolbar.items.length-1;
        while (toolbar.items[i]) {
            var item = toolbar.items[i];
            var elem = Util.getElem(item.name);
            logoX = Util.getPosX(elem);
            var breakAfterThisItem = false;
            if (logoX+tolerance < Util.getWindowWidth()) {
                if (i == toolbar.items.length - 1 && item instanceof ToolBarSeparator) {
                    break; //stop hiding items
                }
                else {
                    breakAfterThisItem = true;
                }
            }
            Util.setProp(elem,"display","none");
            item.isDisplayed = false;
            if (breakAfterThisItem) break; // stop hiding items
            i--;
        }
        for (i in toolbar.items) {
            var item = toolbar.items[i];
            // add toolbar item to context menu
            if (!item.isDisplayed) {
                if (item instanceof ToolBarButton ||
                    item instanceof ToolBarMultiButton ||
                    item instanceof ToolBarSeparator) {
                    toolbar.moreButton.addItem(item);
                }
            }
        }
        if (toolbar.moreButton.hasItems()) {
            Util.setProp(more,"display","block"); // show more button
        }
    }
}


//------------------------------------------------------------------------------
// ToolBarItem
//------------------------------------------------------------------------------

/**
 * Toolbar item constructor
 * @see #init
 */
function ToolBarItem(/*String*/name) {
    this.init(name);
}

/**
 * initiaizes the toolbar item.
 * @param name the name of the toolbar item
 */
ToolBarItem.prototype.init = function(/*String*/ name) {
    this.name = name;
    this.docroot = "";
    this.cssClass = "";
    this.isDisplayed=true;
}

/**
 * Sets the docroot for the images and other resourcers
 * @param docroot the docroot to set
 */
ToolBarItem.prototype.setDocroot = function(/*String*/docroot) {
    this.docroot = docroot;
}

/**
 * Enables or disables this toolbar item
 * @param enable true if item should be enabled, false otherwise
 */
/*void*/ToolBarItem.prototype.enable  = function(/*boolean*/enable) {
    if (this.isDisabled!=enable) return;
    var elem = Util.getElem(this.name);
    if (!elem) return;
    this.isDisabled=!enable;
    elem.innerHTML = this.getInnerHTML();
}

/**
 * Returns the outer HTML of this item
 * @return the outer HTML of this item
 */
/*String*/ToolBarItem.prototype.getOuterHTML = function() {
    return '<td class="'+this.cssClass+'"><div id="'+this.name+'">'+this.getInnerHTML()+'</div></td>';
}

/**
 * Returns the inner HTML of this item
 * @return the inner HTML of this item
 */
/*String*/ ToolBarItem.prototype.getInnerHTML = function() {
    return '&nbsp;';
}

/**
 * subclasses the given obj from this
 */
ToolBarItem.subClassMe = function(/*Object*/ obj) {
    obj.prototype = new this();
    obj.prototype._init = this.prototype.init;
}

//------------------------------------------------------------------------------
// ToolBarSeparator
//------------------------------------------------------------------------------

/**
 * creates a new toolbar separator item
 * @param name the name of the new toolbar
 */
function ToolBarSeparator(name) {
    this.init(name);
}
ToolBarItem.subClassMe(ToolBarSeparator);

/**
 * initializes this item
 * @param name the name of this item
 */
ToolBarSeparator.prototype.init = function(/*String*/name) {
    this._init(name);
    this.cssClass = ToolBar.barClass;
}

/**
 * @see ToolBarItem#getInnerHTML
 */
/*String*/ ToolBarSeparator.prototype.getInnerHTML = function() {
    return '<span class="'+ToolBar.separatorClass+'">&nbsp;</span>';
}

//------------------------------------------------------------------------------
// ToolBarImage
//------------------------------------------------------------------------------

/**
 * creates a new toolbar image item
 * @param name the name of the new toolbar
 */
function ToolBarImage(/*String*/name, /*String*/href, /*String*/ icon) {
    this.init(name, href, icon);
}
ToolBarItem.subClassMe(ToolBarImage);

/**
 * initializes this item
 * @param name the name of this item
 */
ToolBarImage.prototype.init = function(/*String*/name, /*String*/href, /*String*/ icon) {
    this._init(name);
    this.href = href;
    this.icon = icon;
    this.cssClass = ToolBar.logoClass;
}

/**
 * @see ToolBarItem#getInnerHTML
 */
/*String*/ ToolBarImage.prototype.getInnerHTML = function() {
    return '<a class="'+this.cssClass+'" target="_new" href="'+this.href+'"><img class="'+this.cssClass+'" border="0" id="'+this.name+'" name="'+this.name+'" src="'+this.docroot+this.icon+'" /></a>';
}

//------------------------------------------------------------------------------
// ToolBarSpacer
//------------------------------------------------------------------------------

/**
 * creates a new toolbar spacer item
 * @param name the name of the new toolbar
 */
function ToolBarSpacer(/*String*/name, /*String*/width) {
    this.init(name,width);
}
ToolBarItem.subClassMe(ToolBarSpacer);

/**
 * initializes this spacer
 * @param name the name of this item
 * @param width the width of this item
 */
ToolBarSpacer.prototype.init = function(/*String*/name, /*String*/width) {
    this._init(name);
    this.width = width;
    this.cssClass=ToolBar.spacerClass;
}

/**
 * @see ToolBarItem#getOuterHTML
 */
/*String*/ToolBarSpacer.prototype.getOuterHTML = function() {
    return '<td class="'+this.cssClass+'" width="'+this.width+'"><div id="'+this.name+'" style="width:'+this.width+';">&nbsp;</div></td>';
}


//------------------------------------------------------------------------------
// ToolBarButton
//------------------------------------------------------------------------------

/**
 * creates a new toolbar button item
 * @param name the name of the new toolbar
 */
function ToolBarButton(/*String*/name, /*String*/ icon, /*String*/ iconSmall,
              /*String*/text, /*String*/action, /*String*/alt,
              /*boolean*/enabled) {
    this.init(name,icon,iconSmall,text,action,alt,enabled);
}
ToolBarItem.subClassMe(ToolBarButton);

/**
 * initializes this button item
 * @param name the name of this item
 * @param icon the icon for this button
 * @param text the text of the button
 * @param action the action
 * @param alt the alt-text
 * @param enabled the enabled flag
 */
ToolBarButton.prototype.init = function(/*String*/name, /*String*/ icon,
              /*String*/ menuIcon, /*String*/text,/*String*/action,
              /*String*/alt, /*boolean*/enabled) {
    this._init(name);
    this.icon = icon;
    this.menuIcon = menuIcon;
    this.text = text;
    this.action = action;
    this.alt = alt;
    this.isDisabled=!enabled;
    this.cssClass = ToolBar.barClass;
}

/**
 * @see ToolBarItem#getInnerHTML
 */
ToolBarButton.prototype.getInnerHTML = function() {
    var str = "";
    var href= "javascript:"+this.action;
    var cssClass = ToolBar.buttonClass;
    if (this.isDisabled) cssClass +=" "+ToolBar.disabledClass;
    if (!this.isDisabled) {
        str+='<a title="'+this.alt+'" class="'+cssClass+'" href="'+href+'">';
    } else {
        str+='<span title="'+this.alt+'" class="'+cssClass+'">';
    }
    if (this.icon) {
                str+='<img class="'+cssClass+'" src="'+this.docroot+this.icon+'">';
    }
    else {
        str+='<img class="'+cssClass+' " src="'+this.docroot+'/imgs/0.gif" style="width:1px;margin-right:0px">';
    }
    str+= this.text ? this.text : '&nbsp;';
    if (!this.isDisabled) {
        str+='</a>';
    } else {
        str+='</span>';
    }
    return str;
}

//------------------------------------------------------------------------------
// ToolBarMultiButton
//------------------------------------------------------------------------------

/**
 * creates a new toolbar multi button item
 * @param name the name of the new toolbar
 */
function ToolBarMultiButton(/*String*/name, /*String*/ icon, /*String*/ iconSmall,
              /*String*/text, /*String*/action, /*String*/alt,
              /*boolean*/enabled) {
    this.init(name,icon,iconSmall,text,action,alt,enabled);
}
ToolBarItem.subClassMe(ToolBarMultiButton);

/**
 * initializes this multi button item
 * @param name the name of this item
 * @param icon the icon for this button
 * @param text the text of the button
 * @param action the action
 * @param alt the alt-text
 * @param enabled the enabled flag
 */
ToolBarMultiButton.prototype.init = function(/*String*/name, /*String*/ icon,
              /*String*/ menuIcon, /*String*/text,/*String*/action,
              /*String*/alt, /*boolean*/enabled) {
    this._init(name);
    this.icon = icon;
    this.menuIcon = menuIcon;
    this.text = text;
    this.action = action;
    this.alt = alt;
    //if (!this.text) this.text = this.alt;
    this.isDisabled=!enabled;
    this.contextMenu = ContextMenu.create(this.name+"_menu");
    this.contextMenu.setOffset(0, 37);
    this.cssClass = ToolBar.barClass;
}

/**
 * Adds a new item to this multi button item
 * @param item toolbar item to add
 */
/*void*/ToolBarMultiButton.prototype.addItem = function(/*ToolBarItem*/ item) {
    if (item instanceof ToolBarButton) {
        // button item
        var name = item.name + "_MultiButtonItem"; // to avoid "overwriting" of items in ff
        var menuItem = new ContextMenuButton(name, item.menuIcon,
            item.text, item.action, item.alt, !item.isDisabled);
        this.contextMenu.addItem(menuItem);
    } else if (item instanceof ToolBarMultiButton) {
        // multi button item
        var items = item.getItems();
        for (var i in items) {
            // add sub items to multi button item
            this.contextMenu.addItem(items[i]);
        }
    } else if (item instanceof ToolBarSeparator) {
        // separator item
        var menuItem = new ContextMenuSeparator(item.name);
        this.contextMenu.addItem(menuItem);
    }
}

/**
 * Deletes all items from this multi button item
 */
/*void*/ToolBarMultiButton.prototype.clear = function() {
    this.contextMenu.clear();
}

/**
 * Returns the items of this multi button item
 * @return the items of this multi button item
 */
/*Array*/ToolBarMultiButton.prototype.getItems = function() {
    return this.contextMenu.getItems();
}

/**
 * Determines whether multi button item has items
 * @return true if there are items, false otherwise
 */
/*boolean*/ToolBarMultiButton.prototype.hasItems = function() {
    return this.contextMenu.hasItems();
}

/**
 * Sets the 'enabled' state of the item specified by 'name'
 * @param name the name of the item
 * @param enable if true, enables item; if false, disables it
 */
/*void*/ToolBarMultiButton.prototype.enableItem = function(/*String*/ name, /*boolean*/enable) {
    this.contextMenu.enableItem(name,enable);
}

/**
 * Populates the context menu with its items
 */
/*void*/ToolBarMultiButton.prototype.populate = function() {
    return this.contextMenu.populate();
}

/**
 * @see ToolBarItem#setDocroot
 */
ToolBarMultiButton.prototype.setDocroot = function(/*String*/docroot) {
    this.docroot = docroot;
    this.contextMenu.setDocroot(docroot);
}

/**
 * @see ToolBarItem#getInnerHTML
 */
ToolBarMultiButton.prototype.getInnerHTML = function() {
    var str = "";
    var cssClass = ToolBar.buttonClass;
    if (this.isDisabled) cssClass +=" "+ToolBar.disabledClass;
    var href = "javascript:void(0);";
    var onclick = "ContextMenu.toggleMenu('"+this.contextMenu.name+"',this,event)";
    var onfocus = "this.blur()";
    if (!this.isDisabled) {
        str += '<a class="'+cssClass+'" id="'+this.name+'_sub" href="'+href+'" onclick="'+onclick+'" onfocus="'+onfocus+'" title="'+this.alt+'">';
    } else {
        str += '<span title="'+this.alt+'" class="'+cssClass+'">';
    }
    if (this.icon) {
        str+='<img class="'+cssClass+'" src="'+this.docroot+this.icon+'">';
    }
    else {
        str+='<img class="'+cssClass+' " src="'+this.docroot+'/imgs/0.gif" style="width:1px;margin-right:0px">';
    }
    str+= this.text ? this.text : '&nbsp;';
    if (!this.isDisabled) {
        str+='</a>';
    } else {
        str+='</span>';
    }
    str += this.contextMenu.getOuterHTML();
    return str;
}
