/*
 * Copyright (C) 2018 Adyen N.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.adyen.threeds2.parameters;

import com.adyen.threeds2.exception.InvalidInputException;
import com.adyen.threeds2.util.Preconditions;

import java.util.HashMap;
import java.util.Map;

/**
 * This class represents the configuration parameters that are required by the 3DS SDK for initialization.
 * The App should create a ConfigParameters object and sets the required parameter values.
 * <p>
 * Notice: The {@link com.adyen.threeds2.util.AdyenConfigParameters.Builder} utility method can be used to easily
 * create the configuration parameters required by the 3DS SDK.
 * <p>
 * Created by Ran Haveshush on 24/08/2018.
 */
public final class ConfigParameters {
    private final Map<String, Map<String, String>> mParameters = new HashMap<>();

    /**
     * Adds a configuration parameter either to the specified group or to the default group.
     *
     * @param group      The group to which the configuration parameter is to be added.<br>
     *                   Note: If a group is not specified, then the default group will be used.
     * @param paramName  The name of the configuration parameter.
     * @param paramValue The value of the configuration parameter.
     * @throws InvalidInputException This exception shall be thrown if {@code paramName} is {@code null}.
     */
    public void addParam(final String group, final String paramName, final String paramValue) throws InvalidInputException {
        Preconditions.requireNonNull("paramName", paramName);
        final Map<String, String> parameters = getParametersForGroup(group, true);
        parameters.put(paramName, paramValue);
    }

    /**
     * Returns a configuration parameter’s value either from the specified group or from the default group.
     *
     * @param group     The group from which the configuration parameter’s value is to be returned.<br>
     *                  Note: If the group is {@code null}, then the default group shall be used.
     * @param paramName The name of the configuration parameter.
     * @return The value of the specified configuration parameter.
     * @throws InvalidInputException This exception shall be thrown if {@code paramName} is {@code null}.
     */
    public String getParamValue(final String group, final String paramName) throws InvalidInputException {
        Preconditions.requireNonNull("paramName", paramName);
        final Map<String, String> parameters = getParametersForGroup(group, false);

        return parameters != null ? parameters.get(paramName) : null;
    }

    /**
     * Removes a configuration parameter either from the specified group or from the default group.<br>
     * It returns the name of the parameter that it removes.
     *
     * @param group     The group from which the configuration parameter is to be removed.<br>
     *                  Note: If group is {@code null}, then the default group shall be used.
     * @param paramName The name of the configuration parameter.
     * @return The name of the parameter that it removes.
     * @throws InvalidInputException This exception shall be thrown if {@code paramName} is {@code null}.
     */
    public String removeParam(final String group, final String paramName) throws InvalidInputException {
        Preconditions.requireNonNull("paramName", paramName);
        final Map<String, String> parameters = getParametersForGroup(group, false);

        return parameters != null ? parameters.remove(paramName) : null;
    }

    /**
     * Returns a group of configuration parameters value from the specified group.
     *
     * @param group The group from which the configuration parameters values is to be returned.<br>
     *              Note: If the group is {@code null}, then the default group will be used.
     * @return The value of the specified configuration parameter.
     * @throws InvalidInputException This exception shall be thrown if {@code paramName} is {@code null}.
     */
    public Map<String, String> getGroup(final String group) {
        return getParametersForGroup(group, false);
    }

    private Map<String, String> getParametersForGroup(final String group, final boolean createIfNeeded) {
        Map<String, String> parameters = mParameters.get(group);

        if (createIfNeeded && parameters == null) {
            parameters = new HashMap<>();
            mParameters.put(group, parameters);
        }

        return parameters;
    }
}
