/*
 * Copyright (C) 2018 Adyen N.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.adyen.threeds2.util;

import android.content.Context;
import android.text.TextUtils;

import com.adyen.threeds2.ThreeDS2Service;
import com.adyen.threeds2.customization.UiCustomization;
import com.adyen.threeds2.exception.InvalidInputException;
import com.adyen.threeds2.parameters.ConfigParameters;

import java.util.Arrays;
import java.util.Collection;
import java.util.Set;

/**
 * This utility class helps with creating {@link ConfigParameters} out of given directory server information.
 * <p>
 * Created by timon on 10/09/2018.
 */
public final class AdyenConfigParameters {
    public static final Parameter DIRECTORY_SERVER_ID = new Parameter("threeds2.directoryServer", "id");
    public static final Parameter DIRECTORY_SERVER_PUBLIC_KEY = new Parameter("threeds2.directoryServer", "publicKey");
    public static final Parameter DIRECTORY_SERVER_ROOT_CERTIFICATES = new Parameter("threeds2.directoryServer", "rootCertificates");

    public static final Parameter SECURITY_APP_SIGNATURE = new Parameter("security", "appSignature");
    public static final Parameter SECURITY_TRUSTED_APP_STORES = new Parameter("security", "trustedAppStores");
    public static final Parameter SECURITY_MALICIOUS_APPS = new Parameter("security", "maliciousApps");

    // List of device parameters that should be blocked from being added to the DeviceData
    public static final Parameter DEVICE_PARAMETER_BLOCK_LIST = new Parameter(null, "deviceParameterBlockList");

    private static final String COLLECTION_DELIMITER = ";";

    /**
     * Returns a given parameter's values.
     *
     * @param configParameters A {@link ConfigParameters} represents the config parameters from which the parameter's value needed to be retrieved.
     * @param parameter        A {@link Parameter} represents a given parameter which it's value needed to be retrieved.
     * @return A {@link Collection} represents the {@link Parameter}'s values.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.<br>
     *                               In case the {@code configParameters} parameter is {@code null}.
     */
    public static Collection<String> getParamValues(final ConfigParameters configParameters, final Parameter parameter) throws InvalidInputException {
        final String paramValue = getParamValue(configParameters, parameter);
        if (paramValue == null) {
            return null;
        }

        final String[] paramValues = paramValue.split(COLLECTION_DELIMITER);

        return Arrays.asList(paramValues);
    }

    /**
     * Returns a given parameter's value.
     *
     * @param configParameters A {@link ConfigParameters} represents the config parameters from which the parameter's value needed to be retrieved.
     * @param parameter        A {@link Parameter} represents a given parameter which it's value needed to be retrieved.
     * @return A {@link String} represents the {@link Parameter}'s value.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.<br>
     *                               In case the {@code configParameters} parameter is {@code null}.
     */
    public static String getParamValue(final ConfigParameters configParameters, final Parameter parameter) throws InvalidInputException {
        Preconditions.requireNonNull("configParameters", configParameters);
        Preconditions.requireNonNull("parameter", parameter);

        final String group = parameter.getGroup();
        final String paramName = parameter.getParamName();

        return configParameters.getParamValue(group, paramName);
    }

    static void addParam(final ConfigParameters configParameters, final Parameter parameter, final Collection<String> paramValues) throws
            InvalidInputException {
        Preconditions.requireNonNull("paramValues", paramValues);

        addParam(configParameters, parameter, TextUtils.join(COLLECTION_DELIMITER, paramValues));
    }

    static void addParam(final ConfigParameters configParameters, final Parameter parameter, final String paramValue) throws
            InvalidInputException {
        Preconditions.requireNonNull("configParameters", configParameters);
        Preconditions.requireNonNull("parameter", parameter);
        Preconditions.requireNonEmpty("paramValue", paramValue);

        configParameters.addParam(parameter.getGroup(), parameter.getParamName(), paramValue);
    }

    /**
     * This class is an helper class to build {@link AdyenConfigParameters}.
     */
    public static final class Builder {
        private final String mDirectoryServerId;

        private final String mDirectoryServerPublicKey;

        private final String mDirectoryServerRootCertificates;

        private String mAppSignature;

        private Set<String> mTrustedAppStores;

        private Set<String> mMaliciousApps;

        private Set<String> mDeviceParameterBlockList;

        /**
         * A builder class for the creation of {@link ConfigParameters} required in the 3DS SDK service initialization.<br>
         * The 3DS SDK is initialized with {@link ThreeDS2Service#initialize(Context, ConfigParameters, String, UiCustomization)}.
         *
         * @param directoryServerId               A {@link String} represents the directory server ID.
         * @param directoryServerPublicKey        A {@link String} represents the directory server public key.
         * @param directoryServerRootCertificates A {@link String} represents the directory server root certificates.
         */
        public Builder(
                final String directoryServerId,
                final String directoryServerPublicKey,
                final String directoryServerRootCertificates
        ) {
            mDirectoryServerId = directoryServerId;
            mDirectoryServerPublicKey = directoryServerPublicKey;
            mDirectoryServerRootCertificates = directoryServerRootCertificates;
        }

        /**
         * Pass the App's signature to validate the App authenticity.
         * <p>
         * The 3DS SDK validates the given App's signature with the current one,
         * if the later is not valid, the 3DS SDK registers an SDK integrity security warning.
         * <p>
         * For security reasons don't store the App's signature in the App,
         * instead get it from the server via secure protocol like HTTPS.
         * <p>
         * Notice: after the 3DS SDK initialization, it is possible to get the 3DS SDK registered security warning
         * and act accordingly by calling {@link ThreeDS2Service#getWarnings()}.
         *
         * @param appSignature The App's signing keystore SHA256 fingerprint.<br>
         *                     To get the SHA256 string value (e.g.
         *                     82:0B:1E:A7:67:64:D6:68:CC:4B:A4:8E:12:89:67:7E:E9:9D:7C:A1:5B:7C:CA:FA:EE:C0:09:EC:1A:B7:55:0F).<br>
         *                     Run '{@code keytool -list -v -keystore release.keystore}'
         * @return The {@link Builder}.
         */
        public Builder appSignature(final String appSignature) {
            mAppSignature = appSignature;
            return this;
        }

        /**
         * In case the App is published in other app stores than Google Play Store,
         * pass the trusted app stores.
         * <p>
         * The 3DS SDK validates the App's installation app store source,
         * if the app was installed form an untrusted app store the 3DS SDK registers an SDK integrity security warning.
         * <p>
         * Notice: after the 3DS SDK initialization, it is possible to get the 3DS SDK registered security warning
         * and act accordingly by calling {@link ThreeDS2Service#getWarnings()}.
         *
         * @param trustedAppStores The trusted app stores package names the app is available at, defaults to
         *                         Google Play Store.
         * @return The {@link Builder}.
         */
        public Builder trustedAppStores(final Set<String> trustedAppStores) {
            mTrustedAppStores = trustedAppStores;
            return this;
        }

        /**
         * In case the App should run in a safe environment, pass the malicious apps the App shouldn't work with
         * other than Xposed and Cydia substrate.
         * <p>
         * The 3DS SDK validates malicious apps are not installed on the device,
         * if one of the malicious apps is installed, the 3DS SDK registers an SDK integrity security warning.
         * <p>
         * Notice: after the 3DS SDK initialization, it is possible to get the 3DS SDK registered security warning
         * and act accordingly by calling {@link ThreeDS2Service#getWarnings()}.
         *
         * @param maliciousApps The malicious apps package names the SDK should validate are not installed on the device.
         * @return The {@link Builder}.
         */
        public Builder maliciousApps(final Set<String> maliciousApps) {
            mMaliciousApps = maliciousApps;
            return this;
        }

        /**
         * In case you do not wish to collect specific device parameters, you may specify these parameters here.
         *
         * @param deviceParameterBlockList The parameters that should not be collected.
         * @return The {@link Builder}.
         */
        public Builder deviceParameterBlockList(final Set<String> deviceParameterBlockList) {
            mDeviceParameterBlockList = deviceParameterBlockList;
            return this;
        }

        /**
         * Creates the {@link ConfigParameters} required in the 3DS SDK service initialization.<br>
         * The 3DS SDK is initialized with {@link ThreeDS2Service#initialize(Context, ConfigParameters, String, UiCustomization)}.
         *
         * @return The {@link ConfigParameters} required in the 3DS SDK service initialization.
         * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.<br>
         *                               In case the {@code directoryServerId} or {@code directoryServerPublicKey} are empty or
         *                               {@code null}.
         */
        public ConfigParameters build() throws InvalidInputException {
            Preconditions.requireNonEmpty("directoryServerId", mDirectoryServerId);
            Preconditions.requireNonEmpty("directoryServerPublicKey", mDirectoryServerPublicKey);

            final ConfigParameters configParameters = new ConfigParameters();

            addParam(configParameters, DIRECTORY_SERVER_ID, mDirectoryServerId);
            addParam(configParameters, DIRECTORY_SERVER_PUBLIC_KEY, mDirectoryServerPublicKey);

            if (mDirectoryServerRootCertificates != null) {
                addParam(configParameters, DIRECTORY_SERVER_ROOT_CERTIFICATES, mDirectoryServerRootCertificates);
            }

            if (mAppSignature != null) {
                addParam(configParameters, SECURITY_APP_SIGNATURE, mAppSignature);
            }

            if (mTrustedAppStores != null) {
                addParam(configParameters, SECURITY_TRUSTED_APP_STORES, mTrustedAppStores);
            }

            if (mMaliciousApps != null) {
                addParam(configParameters, SECURITY_MALICIOUS_APPS, mMaliciousApps);
            }

            if (mDeviceParameterBlockList != null) {
                addParam(configParameters, DEVICE_PARAMETER_BLOCK_LIST, mDeviceParameterBlockList);
            }

            return configParameters;
        }
    }

    /**
     * This class represents a single {@link AdyenConfigParameters}'s parameter,
     * the parameter is defined by a group & parameter name,
     * to add the parameter to the default global group pass a {@code null} as a group name.
     */
    private static final class Parameter {
        private final String mGroup;

        private final String mParamName;

        Parameter(final String group, final String paramName) {
            mGroup = group;
            mParamName = paramName;
        }

        String getGroup() {
            return mGroup;
        }

        String getParamName() {
            return mParamName;
        }
    }

    private AdyenConfigParameters() {
        throw new IllegalStateException("No instances.");
    }
}
