package com.adyen.threeds2

/**
 * A callback object that implements this interface will receive a challenge status notification
 * from the 3DS SDK.
 * Depending on the [ChallengeResult] of the challenge process, the App may display a error/success
 * message or redirect the Cardholder to a screen in the App.
 *
 * &nbsp
 *
 * A call to `/authorise3ds2` with the
 * [Completed.transactionStatus][ChallengeResult.Completed.transactionStatus] **is required** when
 * the challenge is completed successfully.
 *
 * &nbsp
 *
 * A call to `/authorise3ds2` with the additional details provided in the
 * [Error.additionalDetails][ChallengeResult.Error.additionalDetails]
 * or [Cancelled.additionalDetails][ChallengeResult.Cancelled.additionalDetails]
 * or [Timeout.additionalDetails][ChallengeResult.Timeout.additionalDetails] **and** the
 * [Error.transactionStatus][ChallengeResult.Error.transactionStatus] or
 * [Cancelled.transactionStatus][ChallengeResult.Cancelled.transactionStatus] or
 * [Timeout.transactionStatus][ChallengeResult.Timeout.transactionStatus] **is required** when the
 * transaction is not successfully completed.
 */
fun interface ChallengeStatusHandler {

    /**
     * This method will be called when:
     * * challenge process is completed successfully.
     * * Cardholder selects the option to cancel the transaction in the challenge screen.
     * * 3DS SDK receives an erroneous message from the ACS.
     * * 3DS SDK encounters errors during the challenge process.
     * * challenge process exceeds the timeout specified in the call for
     * [Transaction.doChallenge].
     *
     * When the [ChallengeResult]  is [Completed][ChallengeResult.Completed] the
     * [transactionStatus][ChallengeResult.Completed.transactionStatus] is made available to be
     * sent to Adyen Server.
     *
     * When the [ChallengeResult] is [Cancelled][ChallengeResult.Cancelled] or
     * [Error][ChallengeResult.Error] or [Timeout][ChallengeResult.Timeout] additionalDetails and
     * the transactionStatus are made available to be sent to Adyen Server.
     *
     * @param result The [ChallengeResult] contains information about the challenge result status.
     */
    fun onCompletion(result: ChallengeResult)
}
