/*
 * Copyright (C) 2018 Adyen N.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.adyen.threeds2;

import android.app.Activity;

import com.adyen.threeds2.exception.InvalidInputException;
import com.adyen.threeds2.parameters.ChallengeParameters;

import kotlin.Deprecated;

/**
 * An instance of this interface holds the {@link AuthenticationRequestParameters}.<br>
 * The App needs to pass these parameters to the 3DS Server which requires those parameters to
 * create {@code AReq} messages and to perform the Challenge Flow.
 * <p>
 * Created by Ran Haveshush on 24/08/2018.
 */
public interface Transaction {
    /**
     * Returns the {@link AuthenticationRequestParameters} parameters.
     * <p>
     * The App should include these authentication request parameters in the message to the 3DS
     * server.
     *
     * @return The {@link AuthenticationRequestParameters} parameters, which contains encrypted
     * device information and 3DS SDK information.
     */
    AuthenticationRequestParameters getAuthenticationRequestParameters();

    /**
     * Initiates the 3DS Challenge Flow process.
     * <p>
     * Note: This method shall be called only when the Challenge Flow is to be applied.
     * <p>
     * When this method is called, control of the App is passed to the 3DS SDK.<br>
     * At any point of time, if the time taken by the challenge process exceeds the timeout value
     * passed then the {@link ChallengeStatusReceiver#timedout()} method will be called.
     *
     * @deprecated Use {@link Transaction#doChallenge(Activity, ChallengeParameters,
     * ChallengeStatusHandler, int)} instead. <br>
     * It uses {@link ChallengeStatusHandler} instead of {@link ChallengeStatusReceiver}, a
     * simpler interface to notify the app with the necessary challenge status.
     *
     * @param currentActivity         The current {@link Activity} instance that invoked this
     *                                method.
     * @param challengeParameters     The {@link ChallengeParameters}, details required by the 3DS
     *                                SDK to conduct the challenge process during the transaction.
     *                                The following details are mandatory:
     *                                <ul>
     *                                <li>3DS Server Transaction ID
     *                                <li>ACS Transaction ID
     *                                <li>ACS Reference Number
     *                                <li>ACS Signed Content
     *                                </ul>
     * @param challengeStatusReceiver The {@link ChallengeStatusReceiver} callback object for
     *                                notifying the App about the challenge status.
     * @param timeOut                 The timeout interval (in minutes) within which the challenge
     *                                process must be completed.<br>
     *                                The minimum timeout interval is 5 minutes.
     * @throws InvalidInputException This exception will be thrown if an input parameter is invalid.
     *                               A timeout interval of less than 5 minutes is also treated as
     *                               invalid input.
     */
    @Deprecated(
         message = "Use 'doChallenge(Activity!, ChallengeParameters!, " +
                 "ChallengeStatusHandler!, int): Unit' method instead."
    )
    void doChallenge(
            Activity currentActivity,
            ChallengeParameters challengeParameters,
            ChallengeStatusReceiver challengeStatusReceiver,
            int timeOut
    ) throws InvalidInputException;

    /**
     * Initiates the 3DS Challenge Flow process.
     * <p>
     * Note: This method shall be called only when the Challenge Flow is to be applied.
     * <p>
     * When this method is called, control of the App is passed to the 3DS SDK.<br>
     * If the time taken by the challenge process exceeds the timeout value passed then the
     * {@link ChallengeStatusHandler#onCompletion(ChallengeResult)}} method will be
     * called with {@link ChallengeResult.Timeout}.
     *
     * @param currentActivity         The current {@link Activity} instance that invoked this method.
     * @param challengeParameters     The {@link ChallengeParameters}, details required by the 3DS
     *                                SDK to conduct the challenge process during the transaction.
     *                                The following details are mandatory:
     *                                <ul>
     *                                <li>3DS Server Transaction ID
     *                                <li>ACS Transaction ID
     *                                <li>ACS Reference Number
     *                                <li>ACS Signed Content
     *                                </ul>
     * @param challengeStatusHandler The {@link ChallengeStatusHandler} callback object for
     *                                notifying the App about the challenge status.
     * @param timeOut                 The timeout interval (in minutes) within which the challenge
     *                                process must be completed.<br>
     *                                The minimum timeout interval is 5 minutes.
     * @throws InvalidInputException This exception will be thrown if an input parameter is invalid.
     *                               A timeout interval of less than 5 minutes is also treated as
     *                               invalid input.
     */

    void doChallenge(
            Activity currentActivity,
            ChallengeParameters challengeParameters,
            ChallengeStatusHandler challengeStatusHandler,
            int timeOut
    ) throws InvalidInputException;
    /**
     * Returns a {@link ProgressDialog} instance (processing screen) that the App may use.
     *
     * @param currentActivity The current {@link Activity} instance that invoked this method.
     * @return The {@link ProgressDialog} instance.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is
     * invalid.
     */
    ProgressDialog getProgressView(Activity currentActivity) throws InvalidInputException;

    /**
     * Cleans up resources that are held by the {@link Transaction} object.<br>
     * Call this method when the transaction is completed.
     */
    void close();
}
