/*
 * Copyright (C) 2018 Adyen N.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.adyen.threeds2.parameters;

import android.content.Context;
import androidx.annotation.NonNull;
import com.adyen.threeds2.BuildConfig;
import com.adyen.threeds2.internal.util.PublicLogger;
import static com.adyen.threeds2.internal.challenge.validator.UrlValidatorKt.isCompleteUrl;
import static com.adyen.threeds2.internal.challenge.validator.UrlValidatorKt.isWebLink;

import kotlin.Deprecated;

/**
 * This class holds the parameters that are required to conduct the challenge process.
 * <p>
 * Note: It is mandatory to set values for these parameters.
 * <p>
 * Created by Ran Haveshush on 24/08/2018.
 */
public final class ChallengeParameters {

    private String mThreeDSServerTransactionID;

    private String mAcsTransactionID;

    private String mAcsRefNumber;

    private String mAcsSignedContent;

    private String mThreeDSRequestorAppURL;

    /**
     * Returns the ThreeDSRequestorAppURL handled by the 3DS2 SDK for automatic continuation of OOB flow.
     *
     * @param context A context to get the app package name.
     * @return The ThreeDSRequestorAppURL expected to be passed in the ChallengeParameters.
     * @deprecated Custom deep links for threeDSRequestorAppUrl are deprecated.
     */
    @Deprecated(message = "Replace by your Android App Link associated with ChallengeActivity.")
    @NonNull
    public static String getEmbeddedRequestorAppURL(final Context context) {
        return BuildConfig.threeDSRedirectScheme + "://" + context.getPackageName();
    }

    /**
     * @return The 3DS server transaction ID.
     */
    public String get3DSServerTransactionID() {
        return mThreeDSServerTransactionID;
    }

    /**
     * Sets the 3DS Server Transaction ID.<br>
     * This ID is a transaction identifier assigned by the 3DS Server to uniquely identify a single transaction.
     *
     * @param threeDSServerTransactionID The transaction identifier assigned by the 3DS Server to uniquely identify a single transaction.
     */
    public void set3DSServerTransactionID(final String threeDSServerTransactionID) {
        mThreeDSServerTransactionID = threeDSServerTransactionID;
    }

    /**
     * @return The ACS server transaction ID.
     */
    public String getAcsTransactionID() {
        return mAcsTransactionID;
    }

    /**
     * Sets the ACS Server Transaction ID.
     *
     * @param acsTransactionID The transaction ID assigned by the ACS to uniquely identify a single transaction.
     */
    public void setAcsTransactionID(final String acsTransactionID) {
        mAcsTransactionID = acsTransactionID;
    }

    /**
     * @return The ACS reference number.
     */
    public String getAcsRefNumber() {
        return mAcsRefNumber;
    }

    /**
     * Sets the ACS Reference Number.
     *
     * @param acsRefNumber The ACS identifier assigned by EMVCo.
     */
    public void setAcsRefNumber(final String acsRefNumber) {
        mAcsRefNumber = acsRefNumber;
    }

    /**
     * @return The ACS signed content.
     */
    public String getAcsSignedContent() {
        return mAcsSignedContent;
    }

    /**
     * Sets the ACS signed content.<br>
     * This content includes the ACS URL, ACS ephemeral public key, and SDK ephemeral public key.
     *
     * @param acsSignedContent The ACS signed content. This data includes the ACS URL, ACS ephemeral public key, and SDK ephemeral public key.
     */
    public void setAcsSignedContent(final String acsSignedContent) {
        mAcsSignedContent = acsSignedContent;
    }

    /**
     * Sets the 3DS Requestor App URL.
     * It is strongly recommended to set url in Android App Link (https) format.
     *
     * @param threeDSRequestorAppURL The 3DS Requestor App URL.
     */
    public void setThreeDSRequestorAppURL(String threeDSRequestorAppURL) {
        // If provided URL is not complete - it will not be set to not break the Challenge flow.
        if (threeDSRequestorAppURL != null && !isCompleteUrl(threeDSRequestorAppURL)) {
            PublicLogger.INSTANCE.e(
                    getClass().getSimpleName(),
                    "Provided threeDSRequestorAppURL is not complete URL."
            );
            return;
        }

        if (!isWebLink(threeDSRequestorAppURL)) {
            PublicLogger.INSTANCE.w(
                    getClass().getSimpleName(),
                    "Provided threeDSRequestorAppURL is not an Android App Link. " +
                            "This may negatively impact your 3DS2 challenge performance. " +
                            "For better performance it is strongly recommended to provide " +
                            "threeDSRequestorAppUrl in Android App Link format."
            );
        }
        mThreeDSRequestorAppURL = threeDSRequestorAppURL;
    }

    /**
     * @return the 3DS Requestor App URL as a string.
     */
    public String getThreeDSRequestorAppURL() {
        return mThreeDSRequestorAppURL;
    }
}
