package com.adyen.threeds2.result

import android.os.Build
import com.adyen.threeds2.BuildConfig
import com.adyen.threeds2.internal.util.Base64
import com.adyen.threeds2.internal.util.DestroyableString
import com.adyen.threeds2.result.models.DeviceIdentifiers
import com.adyen.threeds2.result.models.TransactionIdentifiers
import io.michaelrocks.paranoid.Obfuscate
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonObjectBuilder
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.put
import kotlinx.serialization.json.putJsonObject

internal fun String.toResultCode(): ResultCode {
    return when (this) {
        "101" -> ResultCode.ERROR_FROM_ACS_MESSAGE_RECEIVED_INVALID
        "102" -> ResultCode.ERROR_FROM_ACS_MESSAGE_VERSION_NOT_SUPPORTED
        "201" -> ResultCode.ERROR_FROM_ACS_DATA_ELEMENT_MISSING
        "202" -> ResultCode.ERROR_FROM_ACS_MESSAGE_EXTENSION_MISSING
        "203" -> ResultCode.ERROR_FROM_ACS_DATA_ELEMENT_INVALID_FORMAT
        "204" -> ResultCode.ERROR_FROM_ACS_DUPLICATE_DATA_ELEMENT
        "301" -> ResultCode.ERROR_FROM_ACS_TRANSACTION_ID_NOT_RECOGNIZED
        "302" -> ResultCode.ERROR_FROM_ACS_DATA_DECRYPTION_FAILURE
        "303" -> ResultCode.ERROR_FROM_ACS_ACCESS_DENIED
        "304" -> ResultCode.ERROR_FROM_ACS_ISO_CODE_INVALID
        "305" -> ResultCode.ERROR_FROM_ACS_TRANSACTION_DATA_INVALID
        "402" -> ResultCode.ERROR_FROM_ACS_TRANSACTION_TIMED_OUT
        "403" -> ResultCode.ERROR_FROM_ACS_TRANSIENT_SYSTEM_FAILURE
        "404" -> ResultCode.ERROR_FROM_ACS_PERMANENT_SYSTEM_FAILURE
        "405" -> ResultCode.ERROR_FROM_ACS_SYSTEM_CONNECTION_FAILURE
        else -> ResultCode.ERROR_MESSAGE_FROM_ACS_OTHER
    }
}

internal fun getBase64EncodedAdditionalDetails(
    resultCode: ResultCode,
    errorField: MessageField? = null,
    transactionIdentifiers: TransactionIdentifiers,
    messageVersion: String?
): String {
    val deviceIdentifiers = DeviceIdentifiers(
        platform = "Android",
        platformVersion = Build.VERSION.RELEASE ?: "UNKNOWN",
        model = "${Build.MANUFACTURER} ${Build.MODEL}",
    )

    val additionalDetailsJson = createAdditionalDetailsJson(
        resultCode, errorField, transactionIdentifiers, deviceIdentifiers, messageVersion
    )

    deviceIdentifiers.destroy()

    return Base64.get().encodeToString(additionalDetailsJson.toString())
}

internal fun createAdditionalDetailsJson(
    resultCode: ResultCode,
    errorField: MessageField? = null,
    transactionIdentifiers: TransactionIdentifiers,
    deviceIdentifiers: DeviceIdentifiers,
    messageVersion: String? = null,
): JsonObject {
    return buildJsonObject {
        put(AdditionalDetailsField.ERROR_CODE, resultCode.code)
        putIfNotNull(AdditionalDetailsField.ERROR_FIELD, errorField?.identifier)
        putJsonObject(AdditionalDetailsField.ADDITIONAL_DETAILS.identifier) {
            putIfNotNull(
                AdditionalDetailsField.SDK_TRANSACTION_IDENTIFIER,
                transactionIdentifiers.sdkTransactionId,
            )
            putIfNotNull(
                AdditionalDetailsField.SERVER_TRANSACTION_IDENTIFIER,
                transactionIdentifiers.serverTransactionId,
            )
            putIfNotNull(
                AdditionalDetailsField.ACS_TRANSACTION_IDENTIFIER,
                transactionIdentifiers.acsTransactionId,
            )
            putIfNotNull(
                AdditionalDetailsField.ACS_REFERENCE_NUMBER,
                transactionIdentifiers.acsReferenceNumber,
            )
            putIfNotNull(AdditionalDetailsField.MESSAGE_VERSION, messageVersion)
            put(AdditionalDetailsField.SDK_VERSION, BuildConfig.VERSION_NAME)
            put(AdditionalDetailsField.PLATFORM, deviceIdentifiers.platform)
            put(AdditionalDetailsField.PLATFORM_VERSION, deviceIdentifiers.platformVersion)
            put(AdditionalDetailsField.DEVICE_MODEL, deviceIdentifiers.model)
        }
        put(AdditionalDetailsField.VERSION, "1.0")
    }
}

private fun JsonObjectBuilder.putIfNotNull(key: AdditionalDetailsField, value: String?) =
    value?.let { put(key.identifier, value) }

private fun JsonObjectBuilder.put(key: AdditionalDetailsField, value: String?) =
    value?.let { put(key.identifier, value) }

@Obfuscate
internal enum class AdditionalDetailsField(val identifier: String) {
    ERROR_CODE("errorCode"),
    ERROR_FIELD("errorField"),
    ADDITIONAL_DETAILS("additionalDetails"),
    SDK_TRANSACTION_IDENTIFIER("sdkTransactionIdentifier"),
    SERVER_TRANSACTION_IDENTIFIER("serverTransactionIdentifier"),
    ACS_TRANSACTION_IDENTIFIER("acsTransactionIdentifier"),
    ACS_REFERENCE_NUMBER("acsReferenceNumber"),
    MESSAGE_VERSION("messageVersion"),
    SDK_VERSION("sdkVersion"),
    PLATFORM("platform"),
    PLATFORM_VERSION("platformVersion"),
    DEVICE_MODEL("deviceModel"),
    VERSION("version"),
}
