package com.adyen.threeds2

/**
 * An object that represents the final outcome of the challenge process.
 * It's values can be the following:
 * * [Completed][ChallengeResult.Completed]: Challenge process is completed successfully.
 * * [Cancelled][ChallengeResult.Cancelled]: The Cardholder selected the option to cancel the
 * transaction.
 * * [Error][ChallengeResult.Error]: An error occurred during the challenge process. These are
 * usually related with the 3DS Protocol.
 * * [Timeout][ChallengeResult.Timeout]: Challenge process exceeded the specified timeout in
 * [Transaction.doChallenge].
 */
sealed interface ChallengeResult {

    /**
     * Challenge process is completed successfully.
     *
     * @param transactionStatus The [Transaction] status required for Adyen to identify the final
     * status of a transaction.
     */
    class Completed(val transactionStatus: String) : ChallengeResult

    /**
     * The Cardholder selected the option to cancel the transaction.
     *
     * @param transactionStatus The [Transaction] status required for Adyen to identify the final
     * status of a transaction.
     * @param additionalDetails Details required for Adyen to identify the final status of the
     * transaction.
     */
    class Cancelled(
        val transactionStatus: String = DEFAULT_ERROR_TRANSACTION_STATUS,
        val additionalDetails: String,
    ) : ChallengeResult

    /**
     * An error occurred during the challenge process.
     * These errors are often related to the 3DS Protocol.
     *
     * @param transactionStatus The [Transaction] status required for Adyen to identify the final
     * status of a transaction.
     * @param additionalDetails Details required for Adyen to identify the final status of the
     * transaction.
     */
    class Error(
        val transactionStatus: String = DEFAULT_ERROR_TRANSACTION_STATUS,
        val additionalDetails: String,
    ) : ChallengeResult

    /**
     * Challenge process exceeded the specified timeout in [Transaction.doChallenge].
     * The timeout has a minimum of 5 minutes.
     *
     * @param transactionStatus The [Transaction] status required for Adyen to identify the final
     * status of a transaction.
     * @param additionalDetails Details required for Adyen to identify the final status of the
     * transaction.
     */
    class Timeout(
        val transactionStatus: String = DEFAULT_ERROR_TRANSACTION_STATUS,
        val additionalDetails: String,
    ) : ChallengeResult
}

private const val DEFAULT_ERROR_TRANSACTION_STATUS = "U"
