/*
 * Copyright (C) 2018 Adyen N.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.adyen.threeds2.customization;

import android.content.Context;
import android.graphics.Color;
import android.graphics.Typeface;

import com.adyen.threeds2.exception.InvalidInputException;
import com.adyen.threeds2.util.Preconditions;

import java.io.Serializable;

/**
 * This class serves as a superclass for the specific customization classes.
 * <p>
 * Created by Ran Haveshush on 24/08/2018.
 */
public class Customization implements Serializable {

    static final int INITIAL_VALUE = -1;

    private String mTextColorCode;

    private String mTextFontName;

    private int mTextFontSize = INITIAL_VALUE;

    public static Integer parseHexColorCode(final String hexColorCode) {
        if (hexColorCode != null && !hexColorCode.isEmpty() && hexColorCode.charAt(0) == '#') {
            try {
                return Color.parseColor(hexColorCode);
            } catch (Exception e) {
                // Do nothing.
            }
        }

        return null;
    }

    public static Typeface parseTypeface(final Context context, final String fontName) {
        if (fontName == null) {
            return null;
        }

        Typeface typeface = null;

        try {
            typeface = Typeface.createFromAsset(context.getAssets(), fontName);
        } catch (Exception e) {
            // Do nothing.
        }

        if (typeface != null) {
            return typeface;
        }

        try {
            typeface = Typeface.createFromFile(fontName);
        } catch (Exception e) {
            // Do nothing.
        }

        if (typeface != null) {
            return typeface;
        }

        try {
            typeface = Typeface.create(fontName, Typeface.NORMAL);
        } catch (Exception e) {
            // Do nothing.
        }

        if (typeface != null) {
            return typeface;
        }

        return null;
    }

    Customization() {
        // Empty constructor.
    }

    /**
     * @return The color code of the text.
     */
    public String getTextColor() {
        return mTextColorCode;
    }

    /**
     * Sets the text color for a UI element.
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setTextColor(final String hexColorCode) throws InvalidInputException {
        mTextColorCode = requireHexColorCode(hexColorCode);
    }

    /**
     * @return The text font type name.
     */
    public String getTextFontName() {
        return mTextFontName;
    }

    /**
     * Sets the text font type for a UI element.
     *
     * @param fontName The text font type name for the UI element.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setTextFontName(final String fontName) throws InvalidInputException {
        mTextFontName = requireNonEmpty("fontName", fontName);
    }

    /**
     * @return The text font size.
     */
    public int getTextFontSize() {
        return mTextFontSize;
    }

    /**
     * Sets the text font size for a UI element.
     *
     * @param fontSize The text font size for the UI element.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setTextFontSize(final int fontSize) throws InvalidInputException {
        mTextFontSize = requireNonNegative("fontSize", fontSize);
    }

    Integer requireNonNegative(final String name, final int value) throws InvalidInputException {
        Preconditions.requireNonNegative(name, value);
        return value;
    }

    String requireNonEmpty(final String name, final String value) throws InvalidInputException {
        Preconditions.requireNonEmpty(name, value);
        return value;
    }

    String requireHexColorCode(final String hexColorCode) throws InvalidInputException {
        final Integer color = parseHexColorCode(hexColorCode);
        Preconditions.requireNonNull("hexColorCode", color);
        return hexColorCode;
    }
}
