/*
 * Copyright (C) 2018 Adyen N.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.adyen.threeds2.customization;

import com.adyen.threeds2.exception.InvalidInputException;

/**
 * This class provides methods for the App to pass label customization parameters to the 3DS SDK.
 * <p>
 * Created by Ran Haveshush on 24/08/2018.
 */
public final class LabelCustomization extends Customization {

    private String mHeadingTextColorCode;

    private String mHeadingTextFontName;

    private int mHeadingTextFontSize = INITIAL_VALUE;

    private String mInputLabelTextColorCode;

    private String mInputLabelTextFontName;

    private int mInputLabelTextFontSize = INITIAL_VALUE;

    /**
     * @return The color code of the heading text.
     */
    public String getHeadingTextColor() {
        return mHeadingTextColorCode;
    }

    /**
     * Sets the heading text color.
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setHeadingTextColor(final String hexColorCode) throws InvalidInputException {
        mHeadingTextColorCode = requireHexColorCode(hexColorCode);
    }

    /**
     * @return The heading text font type name.
     */
    public String getHeadingTextFontName() {
        return mHeadingTextFontName;
    }

    /**
     * Sets the heading text font type.
     *
     * @param fontName The heading text font type name.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setHeadingTextFontName(final String fontName) throws InvalidInputException {
        mHeadingTextFontName = requireNonEmpty("fontName", fontName);
    }

    /**
     * @return The heading text font size.
     */
    public int getHeadingTextFontSize() {
        return mHeadingTextFontSize;
    }

    /**
     * Sets the heading text font size.
     *
     * @param fontSize The heading text font size.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setHeadingTextFontSize(final int fontSize) throws InvalidInputException {
        mHeadingTextFontSize = requireNonNegative("fontSize", fontSize);
    }

    /**
     * @return The color code of the input label text.
     */
    public String getInputLabelTextColor() {
        return mInputLabelTextColorCode;
    }

    /**
     * Sets the input label text color.
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setInputLabelTextColor(final String hexColorCode) throws InvalidInputException {
        mInputLabelTextColorCode = requireHexColorCode(hexColorCode);
    }

    /**
     * @return The input label text font type name.
     */
    public String getInputLabelTextFontName() {
        return mInputLabelTextFontName;
    }

    /**
     * Sets the input label text font type.
     *
     * @param fontName The input label text font type name.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setInputLabelTextFontName(final String fontName) throws InvalidInputException {
        mInputLabelTextFontName = requireNonEmpty("fontName", fontName);
    }

    /**
     * @return The input label text font size.
     */
    public int getInputLabelTextFontSize() {
        return mInputLabelTextFontSize;
    }

    /**
     * Sets the input label text font size.
     *
     * @param fontSize The input label text font size.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setInputLabelTextFontSize(final int fontSize) throws InvalidInputException {
        mInputLabelTextFontSize = requireNonNegative("fontSize", fontSize);
    }
}
