/*
 * Copyright (C) 2018 Adyen N.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.adyen.threeds2.customization;

import com.adyen.threeds2.exception.InvalidInputException;

/**
 * This class provides methods for the App to pass selection item customization parameters to the 3DS SDK.
 * <p>
 * Created by Ran Haveshush on 01/10/2018.
 */
public final class SelectionItemCustomization extends Customization {

    private String mSelectionIndicatorTintColorCode;

    private String mHighlightedBackgroundColorCode;

    private String mBorderColorCode;

    private int mBorderWidth = INITIAL_VALUE;

    /**
     * @return The tint color code of the selection indicator.
     */
    public String getSelectionIndicatorTintColor() {
        return mSelectionIndicatorTintColorCode;
    }

    /**
     * Sets the tint color of the selection indicator.
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setSelectionIndicatorTintColor(final String hexColorCode) throws InvalidInputException {
        mSelectionIndicatorTintColorCode = requireHexColorCode(hexColorCode);
    }

    /**
     * @return The highlighted background color code of the selection item.
     */
    public String getHighlightedBackgroundColor() {
        return mHighlightedBackgroundColorCode;
    }

    /**
     * Sets the highlighted background color of the selection item.
     *
     * @param hexColorCode {@link String} represents the color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setHighlightedBackgroundColor(final String hexColorCode) throws InvalidInputException {
        mHighlightedBackgroundColorCode = requireHexColorCode(hexColorCode);
    }

    /**
     * @return The hex color code of the border.
     */
    public String getBorderColor() {
        return mBorderColorCode;
    }

    /**
     * Sets the color for the border.
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setBorderColor(final String hexColorCode) throws InvalidInputException {
        mBorderColorCode = requireHexColorCode(hexColorCode);
    }

    /**
     * @return The width of the border between the selection items.
     */
    public int getBorderWidth() {
        return mBorderWidth;
    }

    /**
     * Sets the width of the border between the selection items.
     *
     * @param borderWidth The width of the border.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setBorderWidth(final int borderWidth) throws InvalidInputException {
        mBorderWidth = requireNonNegative("borderWidth", borderWidth);
    }
}
