/*
 * Copyright (C) 2018 Adyen N.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.adyen.threeds2.customization;

import com.adyen.threeds2.exception.InvalidInputException;

/**
 * This class provides methods for the App to pass expandable info customization parameters to the 3DS SDK.
 * <p>
 * Created by Ran Haveshush on 01/10/2018.
 */
public final class ExpandableInfoCustomization extends Customization {

    private String mHeadingTextColorCode;

    private String mHeadingTextFontName;

    private int mHeadingTextFontSize = INITIAL_VALUE;

    private String mBorderColorCode;

    private int mBorderWidth = INITIAL_VALUE;

    private String mExpandedStateIndicatorColorCode;

    private String mHighlightedBackgroundColorCode;

    /**
     * @return The color code of the heading text.
     */
    public String getHeadingTextColor() {
        return mHeadingTextColorCode;
    }

    /**
     * Sets the heading text color.
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setHeadingTextColor(final String hexColorCode) throws InvalidInputException {
        mHeadingTextColorCode = requireHexColorCode(hexColorCode);
    }

    /**
     * @return The heading text font type name.
     */
    public String getHeadingTextFontName() {
        return mHeadingTextFontName;
    }

    /**
     * Sets the heading text font type.
     *
     * @param fontName The heading text font type name.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setHeadingTextFontName(final String fontName) throws InvalidInputException {
        mHeadingTextFontName = requireNonEmpty("fontName", fontName);
    }

    /**
     * @return The heading text font size.
     */
    public int getHeadingTextFontSize() {
        return mHeadingTextFontSize;
    }

    /**
     * Sets the heading text font size.
     *
     * @param fontSize The heading text font size.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setHeadingTextFontSize(final int fontSize) throws InvalidInputException {
        mHeadingTextFontSize = requireNonNegative("fontSize", fontSize);
    }

    /**
     * @return The hex color code of the border.
     */
    public String getBorderColor() {
        return mBorderColorCode;
    }

    /**
     * Sets the color for the border.
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setBorderColor(final String hexColorCode) throws InvalidInputException {
        mBorderColorCode = requireHexColorCode(hexColorCode);
    }

    /**
     * @return The width of the border between the heading text and the info text.
     */
    public int getBorderWidth() {
        return mBorderWidth;
    }

    /**
     * Sets the width of the border between the heading text and the info text.
     *
     * @param borderWidth The width of the border.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setBorderWidth(final int borderWidth) throws InvalidInputException {
        mBorderWidth = requireNonNegative("borderWidth", borderWidth);
    }

    /**
     * @return The color code of the expand state indicator.
     */
    public String getExpandedStateIndicatorColor() {
        return mExpandedStateIndicatorColorCode;
    }

    /**
     * Sets the color of the expand state indicator.
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setExpandStateIndicatorColor(final String hexColorCode) throws InvalidInputException {
        mExpandedStateIndicatorColorCode = requireHexColorCode(hexColorCode);
    }

    /**
     * @return The highlighted background color code of the heading.
     */
    public String getHighlightedBackgroundColor() {
        return mHighlightedBackgroundColorCode;
    }

    /**
     * Sets the highlighted background color of the heading.
     *
     * @param hexColorCode {@link String} represents the color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setHighlightedBackgroundColor(final String hexColorCode) throws InvalidInputException {
        mHighlightedBackgroundColorCode = requireHexColorCode(hexColorCode);
    }
}
