/*
 * Copyright (C) 2018 Adyen N.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.adyen.threeds2;

import android.app.Application;
import android.content.Context;

import androidx.annotation.NonNull;

import com.adyen.threeds2.customization.UiCustomization;
import com.adyen.threeds2.exception.InvalidInputException;
import com.adyen.threeds2.exception.SDKAlreadyInitializedException;
import com.adyen.threeds2.exception.SDKNotInitializedException;
import com.adyen.threeds2.exception.SDKRuntimeException;
import com.adyen.threeds2.internal.ThreeDS2ServiceInternal;
import com.adyen.threeds2.parameters.ConfigParameters;

import java.util.List;

/**
 * The ThreeDS2Service interface is the main 3DS SDK interface.<br>
 * It provides methods to process 3DS transactions.
 * <p>
 * Use {@link ThreeDS2Service#INSTANCE} to get the ThreeDS2Service instance.
 * <p>
 * Created by Ran Haveshush on 24/08/2018.
 */
public interface ThreeDS2Service {
    /**
     * The {@link ThreeDS2Service} instance.
     */
    ThreeDS2Service INSTANCE = ThreeDS2ServiceInternal.INSTANCE;

    /**
     * Initializes the 3DS SDK service instance.
     * <p>
     * Note: Until the {@link ThreeDS2Service} instance is initialized, it will be unusable.
     * <p>
     * During initialization the following tasks are performed:
     * <ul>
     * <li>Security checks. Retrieved by {@link #getWarnings}.
     * <li>Collection of device information for all versions of the protocol that the SDK supports.
     * </ul>
     * Depending on the App's implementation, a {@link ThreeDS2Service} shall be initialized
     * as a background task either during the App startup or when a transaction is initiated.
     * The state of the instance is maintained until the {@link #cleanup} method is called.
     *
     * @param applicationContext An instance of Android {@link Application} context.
     * @param configParameters   The {@link ConfigParameters}, configuration information that shall be used during initialization.
     * @param locale             (optional) A {@link String} that represents the app’s user interface locale (e.g. "en_US").
     *                           If set to {@code null}, then the default device locale will be used.
     * @param uiCustomization    (optional) The {@link UiCustomization}, UI configuration information that is used to specify the UI layout and theme.
     *                           If set to {@code null}, then the default app's theme will be used.
     * @throws InvalidInputException          This exception will be thrown in any of the following scenarios:
     *                                        <ul>
     *                                        <li>configParameters parameter is null.
     *                                        <li>The value of configParameters, locale, or uiCustomization is invalid.
     *                                        </ul>
     * @throws SDKAlreadyInitializedException This exception will be thrown if the 3DS SDK instance has already been initialized.
     * @throws SDKRuntimeException            This exception will be thrown if an internal error is encountered by the 3DS SDK.
     */
    void initialize(
            Context applicationContext,
            ConfigParameters configParameters,
            String locale,
            UiCustomization uiCustomization
    ) throws InvalidInputException, SDKAlreadyInitializedException, SDKRuntimeException;

    /**
     * Creates an instance of {@link Transaction} through which the App gets the data required to perform the transaction.<br>
     * The App shall call this method for each transaction that is to be processed.
     *
     * @param directoryServerID The Registered Application Provider Identifier (RID) that is unique to the Payment System.
     * @param messageVersion    (optional) The Protocol version according to which the transaction shall be created.
     *                          If empty or {@code null}, then the highest protocol version that the 3DS SDK supports will be used.
     * @return An instance of the {@link Transaction} interface.
     * @throws InvalidInputException      This exception will be thrown if an input parameter is invalid. This also includes an invalid Directory
     *                                    Server ID
     *                                    or a protocol version that the 3DS SDK does not support.
     * @throws SDKNotInitializedException This exception will be thrown if the 3DS SDK instance has not been initialized.
     * @throws SDKRuntimeException        This exception will be thrown if an internal error is encountered by the 3DS SDK.
     */
    Transaction createTransaction(
            String directoryServerID,
            @NonNull String messageVersion
    ) throws InvalidInputException, SDKNotInitializedException, SDKRuntimeException;

    /**
     * Frees up resources that are used by the 3DS SDK. It is called only once during a single App session.
     *
     * @param applicationContext An instance of Android {@link Application} context.
     * @throws SDKNotInitializedException This exception will be thrown if the 3DS SDK instance has not been initialized.
     */
    void cleanup(Context applicationContext) throws SDKNotInitializedException;

    /**
     * Returns the version of the 3DS SDK that is integrated with the App.
     *
     * @return The version of the 3DS SDK.
     */
    String getSDKVersion();

    /**
     * Returns the security checks warnings produced by the 3DS SDK during initialization.
     *
     * @return A {@link List} of {@link Warning}s.
     */
    List<Warning> getWarnings();
}
