/*
 * Copyright (C) 2018 Adyen N.V.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.adyen.threeds2.customization;

import android.app.Activity;
import android.widget.EditText;

import com.adyen.threeds2.exception.InvalidInputException;
import com.adyen.threeds2.util.Preconditions;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This class provides the functionality required to customize the 3DS SDK UI elements.<br>
 * An object of this class holds various UI-related parameters.
 * <p>
 * Created by Ran Haveshush on 24/08/2018.
 */
public final class UiCustomization {
    /**
     * This enum represents the {@link ButtonCustomization} button's type.
     */
    public enum ButtonType {
        VERIFY, CONTINUE, NEXT, CANCEL, RESEND
    }

    private final Map<ButtonType, ButtonCustomization> mButtonTypeCustomizationMap = new HashMap<>();

    private final Map<Class<? extends Customization>, Customization> mCustomizationMap = new HashMap<>();

    /**
     * Returns a {@link ButtonCustomization} instance for a specified button type {@link ButtonType}.
     *
     * @param buttonType A {@link ButtonType} that represents a specific button type to retrieve.
     * @return A {@link ButtonCustomization} for the specified {@link ButtonType}.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public ButtonCustomization getButtonCustomization(final ButtonType buttonType) throws InvalidInputException {
        Preconditions.requireNonNull("buttonType", buttonType);
        return getOrCreateButtonCustomization(buttonType);
    }

    /**
     * Sets the {@link ButtonCustomization} for a specific {@link ButtonType}.<br>
     * The 3DS SDK uses this object for customizing buttons.
     *
     * @param buttonCustomization A {@link ButtonCustomization} that represents button customization object.
     * @param buttonType          A {@link ButtonType} that represents button type.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setButtonCustomization(final ButtonCustomization buttonCustomization, final ButtonType buttonType) throws InvalidInputException {
        Preconditions.requireNonNull("buttonType", buttonType);
        mButtonTypeCustomizationMap.put(buttonType, buttonCustomization);
    }

    /**
     * @return A {@link ScreenCustomization} that represents the screen customization object.
     */
    public ScreenCustomization getScreenCustomization() {
        return getOrCreateCustomization(ScreenCustomization.class);
    }

    /**
     * Sets the screen customization.<br>
     * The 3DS SDK uses this object for customizing the challenge {@link Activity} screen.
     *
     * @param screenCustomization A {@link ScreenCustomization} that represents the screen customization object.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setScreenCustomization(final ScreenCustomization screenCustomization) throws InvalidInputException {
        Preconditions.requireNonNull("screenCustomization", screenCustomization);
        mCustomizationMap.put(ScreenCustomization.class, screenCustomization);
    }

    /**
     * @return A {@link ToolbarCustomization} that represents the toolbar customization object.
     */
    public ToolbarCustomization getToolbarCustomization() {
        return getOrCreateCustomization(ToolbarCustomization.class);
    }

    /**
     * Sets the toolbar customization.<br>
     * The 3DS SDK uses this object for customizing toolbars.
     *
     * @param toolbarCustomization A {@link ToolbarCustomization} that represents the toolbar customization object.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setToolbarCustomization(final ToolbarCustomization toolbarCustomization) throws InvalidInputException {
        Preconditions.requireNonNull("toolbarCustomization", toolbarCustomization);
        mCustomizationMap.put(ToolbarCustomization.class, toolbarCustomization);
    }

    /**
     * @return A {@link LabelCustomization} that represents the label customization object.
     */
    public LabelCustomization getLabelCustomization() {
        return getOrCreateCustomization(LabelCustomization.class);
    }

    /**
     * The setLabelCustomization method shall accept a LabelCustomization object. The 3DS SDK uses this object for customizing labels.
     *
     * @param labelCustomization A {@link LabelCustomization} that represents the label customization object.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setLabelCustomization(final LabelCustomization labelCustomization) throws InvalidInputException {
        Preconditions.requireNonNull("labelCustomization", labelCustomization);
        mCustomizationMap.put(LabelCustomization.class, labelCustomization);
    }

    /**
     * @return A {@link TextBoxCustomization} that represents the text box ({@link EditText}) customization object.
     */
    public TextBoxCustomization getTextBoxCustomization() {
        return getOrCreateCustomization(TextBoxCustomization.class);
    }

    /**
     * Sets the text boxes ({@link EditText}) customization.<br>
     * The 3DS SDK uses this object for customizing text boxes ({@link EditText}).
     *
     * @param textBoxCustomization A {@link TextBoxCustomization} that represents the text box customization object.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setTextBoxCustomization(final TextBoxCustomization textBoxCustomization) throws InvalidInputException {
        Preconditions.requireNonNull("textBoxCustomization", textBoxCustomization);
        mCustomizationMap.put(TextBoxCustomization.class, textBoxCustomization);
    }

    /**
     * @return A {@link SelectionItemCustomization} that represents the selection item customization object.
     */
    public SelectionItemCustomization getSelectionItemCustomization() {
        return getOrCreateCustomization(SelectionItemCustomization.class);
    }

    /**
     * Sets the selection items customization.<br>
     * The 3DS SDK uses this object for customizing selection items.
     *
     * @param selectionItemCustomization A {@link SelectionItemCustomization} that represents the selection item customization object.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setSelectionItemCustomization(final SelectionItemCustomization selectionItemCustomization) throws InvalidInputException {
        Preconditions.requireNonNull("selectionItemCustomization", selectionItemCustomization);
        mCustomizationMap.put(SelectionItemCustomization.class, selectionItemCustomization);
    }

    /**
     * @return A {@link ExpandableInfoCustomization} that represents the expandable info customization object.
     */
    public ExpandableInfoCustomization getExpandableInfoCustomization() {
        return getOrCreateCustomization(ExpandableInfoCustomization.class);
    }

    /**
     * Sets the expandable infos customization.<br>
     * The 3DS SDK uses this object for customizing expandable infos.
     *
     * @param expandableInfoCustomization A {@link ExpandableInfoCustomization} that represents the expandable info customization object.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setExpandableInfoCustomization(final ExpandableInfoCustomization expandableInfoCustomization) throws InvalidInputException {
        Preconditions.requireNonNull("expandableInfoCustomization", expandableInfoCustomization);
        mCustomizationMap.put(ExpandableInfoCustomization.class, expandableInfoCustomization);
    }

    /**
     * This utility method sets the toolbar title by setting the {@link ToolbarCustomization} header text.
     *
     * @param title The toolbar title.
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    public void setToolbarTitle(final String title) {
        Preconditions.requireNonEmpty("title", title);

        final ToolbarCustomization toolbarCustomization = getOrCreateCustomization(ToolbarCustomization.class);
        toolbarCustomization.setHeaderText(title);
    }

    /**
     * This utility method sets the status bar color by setting the {@link ScreenCustomization} status bar color.
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    @SuppressWarnings("PMD.AvoidDuplicateLiterals")
    public void setStatusBarColor(final String hexColorCode) throws InvalidInputException {
        Preconditions.requireNonEmpty("hexColorCode", hexColorCode);

        final ScreenCustomization screenCustomization = getOrCreateCustomization(ScreenCustomization.class);
        screenCustomization.setStatusBarColor(hexColorCode);
    }

    /**
     * This utility method sets the challenge {@link Activity}'s background color by setting the {@link ScreenCustomization} background color.
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    @SuppressWarnings("PMD.AvoidDuplicateLiterals")
    public void setScreenBackgroundColor(final String hexColorCode) throws InvalidInputException {
        Preconditions.requireNonEmpty("hexColorCode", hexColorCode);

        final ScreenCustomization screenCustomization = getOrCreateCustomization(ScreenCustomization.class);
        screenCustomization.setBackgroundColor(hexColorCode);
    }

    /**
     * This utility method sets the text color for all of the specific customizations.
     * <ul>
     * <li>{@link ScreenCustomization}
     * <li>{@link ToolbarCustomization}
     * <li>{@link LabelCustomization}
     * <li>{@link TextBoxCustomization}
     * <li>{@link SelectionItemCustomization}
     * <li>{@link ExpandableInfoCustomization}
     * <li>{@link ButtonCustomization}
     * </ul>
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    @SuppressWarnings("PMD.AvoidDuplicateLiterals")
    public void setTextColor(final String hexColorCode) throws InvalidInputException {
        Preconditions.requireNonEmpty("hexColorCode", hexColorCode);

        final ScreenCustomization screenCustomization = getOrCreateCustomization(ScreenCustomization.class);
        screenCustomization.setTextColor(hexColorCode);

        final ToolbarCustomization toolbarCustomization = getOrCreateCustomization(ToolbarCustomization.class);
        toolbarCustomization.setTextColor(hexColorCode);

        final ButtonCustomization buttonCustomization = getOrCreateButtonCustomization(ButtonType.CANCEL);
        buttonCustomization.setTextColor(hexColorCode);

        final LabelCustomization labelCustomization = getOrCreateCustomization(LabelCustomization.class);
        labelCustomization.setTextColor(hexColorCode);
        labelCustomization.setHeadingTextColor(hexColorCode);
        labelCustomization.setInputLabelTextColor(hexColorCode);

        final TextBoxCustomization textBoxCustomization = getOrCreateCustomization(TextBoxCustomization.class);
        textBoxCustomization.setTextColor(hexColorCode);

        final SelectionItemCustomization selectionItemCustomization = getOrCreateCustomization(SelectionItemCustomization.class);
        selectionItemCustomization.setTextColor(hexColorCode);

        final ExpandableInfoCustomization expandableInfoCustomization = getOrCreateCustomization(ExpandableInfoCustomization.class);
        expandableInfoCustomization.setTextColor(hexColorCode);
        expandableInfoCustomization.setHeadingTextColor(hexColorCode);
        expandableInfoCustomization.setExpandStateIndicatorColor(hexColorCode);
    }

    /**
     * This utility method sets the border color for all of the specific customizations.
     * <ul>
     * <li>{@link TextBoxCustomization}
     * <li>{@link SelectionItemCustomization}
     * <li>{@link ExpandableInfoCustomization}
     * </ul>
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    @SuppressWarnings("PMD.AvoidDuplicateLiterals")
    public void setBorderColor(final String hexColorCode) throws InvalidInputException {
        Preconditions.requireNonEmpty("hexColorCode", hexColorCode);

        final TextBoxCustomization textBoxCustomization = getOrCreateCustomization(TextBoxCustomization.class);
        textBoxCustomization.setBorderColor(hexColorCode);

        final SelectionItemCustomization selectionItemCustomization = getOrCreateCustomization(SelectionItemCustomization.class);
        selectionItemCustomization.setBorderColor(hexColorCode);

        final ExpandableInfoCustomization expandableInfoCustomization = getOrCreateCustomization(ExpandableInfoCustomization.class);
        expandableInfoCustomization.setBorderColor(hexColorCode);
    }

    /**
     * This utility method sets the tint color for all of the specific customizations.
     * <ul>
     * <li>{@link ToolbarCustomization}
     * <li>{@link SelectionItemCustomization}
     * <li>{@link ButtonCustomization} for the borderless buttons {@link ButtonType#RESEND} and {@link ButtonType#CANCEL}
     * </ul>
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    @SuppressWarnings("PMD.AvoidDuplicateLiterals")
    public void setTintColor(final String hexColorCode) throws InvalidInputException {
        Preconditions.requireNonEmpty("hexColorCode", hexColorCode);

        final ToolbarCustomization toolbarCustomization = getOrCreateCustomization(ToolbarCustomization.class);
        toolbarCustomization.setBackgroundColor(hexColorCode);

        final SelectionItemCustomization selectionItemCustomization = getOrCreateCustomization(SelectionItemCustomization.class);
        selectionItemCustomization.setSelectionIndicatorTintColor(hexColorCode);

        for (final ButtonType buttonType : ButtonType.values()) {
            final ButtonCustomization buttonCustomization = getOrCreateButtonCustomization(buttonType);

            switch (buttonType) {
                case CANCEL:
                    break;
                case RESEND:
                    buttonCustomization.setTextColor(hexColorCode);
                    break;
                default:
                    buttonCustomization.setBackgroundColor(hexColorCode);
                    break;
            }
        }
    }

    /**
     * This utility method sets the highlighted background color for all of the specific customizations.
     * <ul>
     * <li>{@link SelectionItemCustomization}
     * <li>{@link ExpandableInfoCustomization}
     * <li>{@link ButtonCustomization} for the borderless buttons {@link ButtonType#RESEND} and {@link ButtonType#CANCEL}
     * </ul>
     *
     * @param hexColorCode The color code in Hex format. For example "#999999".
     * @throws InvalidInputException This exception shall be thrown if an input parameter is invalid.
     */
    @SuppressWarnings("PMD.AvoidDuplicateLiterals")
    public void setHighlightedBackgroundColor(final String hexColorCode) throws InvalidInputException {
        Preconditions.requireNonEmpty("hexColorCode", hexColorCode);

        final SelectionItemCustomization selectionItemCustomization = getOrCreateCustomization(SelectionItemCustomization.class);
        selectionItemCustomization.setHighlightedBackgroundColor(hexColorCode);

        final ExpandableInfoCustomization expandableInfoCustomization = getOrCreateCustomization(ExpandableInfoCustomization.class);
        expandableInfoCustomization.setHighlightedBackgroundColor(hexColorCode);

        final List<ButtonType> borderlessButtonTypes = Arrays.asList(ButtonType.CANCEL, ButtonType.RESEND);

        for (final ButtonType buttonType : borderlessButtonTypes) {
            final ButtonCustomization buttonCustomization = getOrCreateButtonCustomization(buttonType);
            buttonCustomization.setBackgroundColor(hexColorCode);
        }
    }

    private ButtonCustomization getOrCreateButtonCustomization(final ButtonType buttonType) {
        ButtonCustomization buttonCustomization = mButtonTypeCustomizationMap.get(buttonType);

        if (buttonCustomization == null) {
            buttonCustomization = new ButtonCustomization();
            mButtonTypeCustomizationMap.put(buttonType, buttonCustomization);
        }

        return buttonCustomization;
    }

    @SuppressWarnings("unchecked")
    private <T extends Customization> T getOrCreateCustomization(final Class<T> clazz) {
        final Customization customization = mCustomizationMap.get(clazz);

        if (customization == null) {
            try {
                final T t = clazz.newInstance();
                mCustomizationMap.put(clazz, t);
                return t;
            } catch (InstantiationException e) {
                throw new RuntimeException("Could not instantiate " + clazz.getSimpleName(), e);
            } catch (IllegalAccessException e) {
                throw new RuntimeException("Could not access constructor of " + clazz.getSimpleName(), e);
            }
        }

        return (T) customization;
    }
}
