package com.aeontronix.commons.file;

import com.aeontronix.commons.file.builder.FilesBuilder;
import com.aeontronix.commons.io.IOUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.*;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class FileUtils {
    public static final Pattern defaultPathPattern = compilePathSplitPattern(File.separatorChar);

    public static FilesBuilder buildDir() {
        return new FilesBuilder();
    }

    public static void mkdir(File file) throws IOException {
        if( file.exists() && file.isDirectory() ) {
            return;
        }
        if( ! file.mkdir() ) {
            throw new IOException("Unable to create directory: "+file.getPath());
        }
    }

    public static void mkdirs(File file) throws IOException {
        if( file.exists() && file.isDirectory() ) {
            return;
        }
        if( ! file.mkdirs() ) {
            throw new IOException("Unable to create directory: "+file.getPath());
        }
    }

    public static void copy( File src, File dst) throws IOException {
        if( src.isDirectory() ) {
            copyDirectory(src,dst);
        } else {
            copyFile(src, dst);
        }
    }

    public static void copyFile(File src, File dst ) throws IOException {
        FileInputStream is = null;
        FileOutputStream os = null;
        try {
            is = new FileInputStream(src);
            os = new FileOutputStream(dst);
            IOUtils.copy(is,os);
        } finally {
            IOUtils.close(is,os);
        }
    }

    /**
     * Copy the files in the specified directory to another directory
     * @param dir Source directory
     * @param destination Destination Directory
     */
    public static void copyDirectory(File dir, File destination) throws IOException {
        for (File file : listFileInDir(dir)) {
            File destFile = new File(destination, file.getName());
            if( file.isDirectory() ) {
                mkdir(destFile);
                copyDirectory(file,destFile);
            } else {
                copyFile(file,destFile);
            }
        }
    }

    public static byte[] toByteArray(File file) throws IOException {
        FileInputStream is = new FileInputStream(file);
        try {
            return IOUtils.toByteArray(is);
        } finally {
            IOUtils.close(is);
        }
    }

    public static void checkFileIsDirectory(File... files) throws IOException {
        for (File file : files) {
            if( ! file.exists() ) {
                throw new IOException("File "+file.getPath()+" doesn't exist");
            }
            if( ! file.isDirectory() ) {
                throw new IOException("File "+file.getPath()+" isn't a directory");
            }
        }
    }

    public static File[] listFileInDir(File directory) throws IOException {
        File[] files = directory.listFiles();
        if( files == null ) {
            throw new IOException("Unable to list files in directory: "+directory.getPath());
        }
        return files;
    }

    /**
     * List all files/directories in a directory
     *
     * @param directory    Directory to list files in
     * @param recursive    If the search should be done recursively
     * @param includeFiles If files should be included in the list
     * @param includeDirs  If directories should be included in the list
     * @return List of files/directories
     * @throws IOException If an error listing the file occurs
     */
    public static Set<String> listAllFilesNames(File directory, boolean recursive, boolean includeFiles, boolean includeDirs) throws IOException {
        HashSet<String> filepaths = new HashSet<String>();
        recursiveFileNameList(filepaths, directory, null, recursive, includeFiles, includeDirs);
        return filepaths;
    }

    private static void recursiveFileNameList(HashSet<String> filepaths, File file, String path, boolean recursive, boolean includeFiles, boolean includeDirs) throws IOException {
        if (file.isDirectory()) {
            if (path == null || recursive) {
                File[] files = file.listFiles();
                if (files == null) {
                    throw new IOException("Unable to list files in directory: " + file.getPath());
                }
                for (File f : files) {
                    recursiveFileNameList(filepaths, f, path != null ? path + File.separator + f.getName() : f.getName(), recursive, includeFiles, includeDirs);
                }
            }
            if (path != null && includeDirs) {
                filepaths.add(path);
            }
        } else {
            if (includeFiles) {
                filepaths.add(path);
            }
        }
    }

    public static String toString(File file) throws IOException {
        return toString(file, "UTF-8");
    }

    public static String toString(File file, String encoding) throws IOException {
        StringWriter buffer = new StringWriter();
        InputStreamReader fileReader = null;
        try {
            fileReader = new InputStreamReader(new FileInputStream(file), encoding);
            IOUtils.copy(fileReader, buffer);
            return buffer.toString();
        } finally {
            IOUtils.close(buffer);
            IOUtils.close(fileReader);
        }
    }

    /**
     * Write string to a file using UTF-8 encoding
     *
     * @param file File to write data to
     * @param text Text to write
     */
    public static void write(File file, String text) throws IOException {
        write(file, text, "UTF-8");
    }

    /**
     * Write string to a file
     *
     * @param file     File to write text to
     * @param text     Text to write
     * @param encoding Character encoding
     */
    public static void write(File file, String text, String encoding) throws IOException {
        FileOutputStream fos = new FileOutputStream(file);
        try {
            fos.write(text.getBytes(encoding));
        } finally {
            IOUtils.close(fos);
        }
    }

    /**
     * Write data to a file
     *
     * @param file File to write data to
     * @param data Data to write
     */
    public static void write(File file, byte[] data) throws IOException {
        FileOutputStream fos = new FileOutputStream(file);
        try {
            fos.write(data);
        } finally {
            IOUtils.close(fos);
        }
    }

    /**
     * Same as invoking {@link #delete(boolean, File...)} with deleteOnExit set to true)}
     *
     * @param file File to delete
     */
    public static void delete(File... file) throws IOException {
        delete(true, file);
    }

    /**
     * Delete files/directories
     *
     * @param files        files or directories to delete
     * @param deleteOnExit If true and deleting a file fails, it will schedule the file to delete on JVM exit rather than throw an exception
     */
    public static void delete(boolean deleteOnExit, File... files) throws IOException {
        for (File file : files) {
            if (file != null && file.exists()) {
                if (file.isDirectory()) {
                    File[] childrens = file.listFiles();
                    if (childrens == null) {
                        throw new IOException("Unable to list files in " + file.getPath());
                    }
                    for (File f : childrens) {
                        delete(deleteOnExit, f);
                    }
                }
                if (!file.delete()) {
                    if (deleteOnExit) {
                        file.deleteOnExit();
                    } else {
                        throw new IOException("Unable to delete: " + file.getPath());
                    }
                }
            }
        }
    }

    public static SplitPath splitFileNameFromParentPath(String fullPath) {
        return splitFileNameFromParentPath(fullPath,File.separatorChar);
    }

    public static SplitPath splitFileNameFromParentPath(String fullPath, char fileSeparator) {
        final Pattern p;
        if( fileSeparator == File.separatorChar ) {
            p = defaultPathPattern;
        } else {
            p = compilePathSplitPattern(fileSeparator);
        }
        Matcher m = p.matcher(fullPath);
        if( ! m.find() ) {
            throw new IllegalArgumentException("Path pattern cannot be parsed: "+fullPath);
        }
        return new SplitPath(m.group(1),m.group(2));
    }

    public static Pattern compilePathSplitPattern(char fileSeparator) {
        return Pattern.compile("(?:(.*)"+(fileSeparator == '\\' ? "\\\\" : fileSeparator)+")?(.*)");
    }

    public static String getFileExtension(File file) {
        final String fn = file.getName();
        final int i = fn.lastIndexOf('.');
        if( i == -1 ) {
            return null;
        }
        return fn.substring(i + 1);
    }

    public static class SplitPath {
        private String parentPath;
        private String filename;

        public SplitPath(@Nullable String parentPath, @NotNull String filename) {
            this.parentPath = parentPath;
            this.filename = filename;
        }

        public String getParentPath() {
            return parentPath;
        }

        public void setParentPath(String parentPath) {
            this.parentPath = parentPath;
        }

        public String getFilename() {
            return filename;
        }

        public void setFilename(String filename) {
            this.filename = filename;
        }

        @Override
        public String toString() {
            return "SplitPath{" +
                    "parentPath='" + parentPath + '\'' +
                    ", filename='" + filename + '\'' +
                    '}';
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (!(o instanceof SplitPath)) return false;

            SplitPath splitPath = (SplitPath) o;

            if (parentPath != null ? !parentPath.equals(splitPath.parentPath) : splitPath.parentPath != null)
                return false;
            return filename.equals(splitPath.filename);
        }

        @Override
        public int hashCode() {
            int result = parentPath != null ? parentPath.hashCode() : 0;
            result = 31 * result + filename.hashCode();
            return result;
        }
    }
}
