package com.aeontronix.anypointsdk;

import com.aeontronix.anypointsdk.accessmanagement.AccessManagementClient;
import com.aeontronix.anypointsdk.amc.AMCClient;
import com.aeontronix.anypointsdk.auth.AnypointAuthenticationHandler;
import com.aeontronix.anypointsdk.auth.AnypointUPWAuthenticationHandler;
import com.aeontronix.anypointsdk.auth.credentials.AnypointCredentials;
import com.aeontronix.anypointsdk.auth.user.UserResponse;
import com.aeontronix.anypointsdk.cloudhub.CloudhubClient;
import com.aeontronix.anypointsdk.exchange.ExchangeClient;
import com.aeontronix.anypointsdk.monitoring.MonitoringClient;
import com.aeontronix.anypointsdk.organization.Environment;
import com.aeontronix.anypointsdk.organization.EnvironmentData;
import com.aeontronix.anypointsdk.organization.Organization;
import com.aeontronix.commons.io.IOUtils;
import com.aeontronix.restclient.*;
import com.aeontronix.restclient.auth.AuthenticationHandler;
import com.aeontronix.restclient.json.JsonConverterJacksonImpl;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.InjectableValues;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;

import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static org.slf4j.LoggerFactory.getLogger;

public class AnypointClient implements AutoCloseable {
    private static final Logger logger = getLogger(AnypointClient.class);
    public static final String PATH_USER = "/accounts/api/me";
    public static final String HEADER_ORG = "X-ANYPNT-ORG-ID";
    public static final String HEADER_ENV = "X-ANYPNT-ENV-ID";
    public static final String ANYPOINT_DEFAULT_URL = "https://anypoint.mulesoft.com";
    private final String anypointUrl;
    private final RESTClient restClient;
    private final RESTClientHost anypointRestClient;
    private CloudhubClient cloudhubClient;
    private AMCClient amcClient;
    private MonitoringClient monitoringClient;
    private AccessManagementClient accessManagementClient;
    private ExchangeClient exchangeClient;

    private AnypointClient(String anypointUrl, RESTClient restClient, RESTClientHost anypointRestClient) {
        this.anypointUrl = anypointUrl;
        this.restClient = restClient;
        this.anypointRestClient = anypointRestClient;
        cloudhubClient = new CloudhubClient(this);
        amcClient = new AMCClient(this);
        monitoringClient = new MonitoringClient(this);
        accessManagementClient = new AccessManagementClient(this);
        exchangeClient = new ExchangeClient(this);
    }

    public String getAnypointUrl() {
        return anypointUrl;
    }

    public RESTClient getRestClient() {
        return restClient;
    }

    public RESTClientHost getAnypointRestClient() {
        return anypointRestClient;
    }

    public CloudhubClient getCloudhubClient() {
        return cloudhubClient;
    }

    public AMCClient getAMCClient() {
        return amcClient;
    }

    public MonitoringClient getMonitoringClient() {
        return monitoringClient;
    }

    public AccessManagementClient getAccessManagementClient() {
        return accessManagementClient;
    }

    public ExchangeClient getExchangeClient() {
        return exchangeClient;
    }

    public void setAccessManagementClient(AccessManagementClient accessManagementClient) {
        this.accessManagementClient = accessManagementClient;
    }

    public static Builder builder() {
        return new Builder();
    }

    public UserResponse getUser() throws RESTException {
        return anypointRestClient.get(PATH_USER, UserResponse.class);
    }

    public List<Organization> findOrganizations() throws RESTException {
        return getUser().getUser().getMemberOfOrganizations().stream().map(o -> new Organization(this, o))
                .collect(Collectors.toList());
    }

    public boolean testAuthentication() throws RESTException {
        try {
            getUser();
            return true;
        } catch (RESTException e) {
            if (e.isStatusCode401()) {
                return false;
            } else {
                throw e;
            }
        }
    }

    @Override
    public void close() throws IOException {
        IOUtils.close(restClient);
    }

    protected void buildRestClient(AnypointAuthenticationHandler authenticationHandler) {
    }

    public Optional<Organization> findOrganization(@NotNull String id) throws RESTException {
        return findOrganizations().stream().filter(v -> id.equals(v.getId())).findFirst();
    }

    public List<Environment> findEnvironments(String orgId) throws RESTException {
        try {
            try (final PaginatedResponse<EnvironmentData> response = getAnypointRestClient().get().path("/accounts/api/organizations/").path(orgId, true)
                    .path("environments").build().executePaginated(EnvironmentData.class, "offset", "limit", "/data")) {
                return response.toList().stream().map(Environment::new).collect(Collectors.toList());
            }
        } catch (IOException e) {
            throw new RESTException(e);
        }
    }

    public static class Builder {
        private String baseUrl = ANYPOINT_DEFAULT_URL;
        private ProxySettings proxySettings;
        private AuthenticationHandler authenticationHandler;
        private RESTClient restClient;
        private RESTClientHost anypointRestClient;

        Builder() {
        }

        public Builder anypointUrl(@NotNull String baseUrl) {
            this.baseUrl = baseUrl;
            return this;
        }

        public Builder credentials(@NotNull AnypointCredentials credentials) {
            return authenticationHandler(credentials.toAuthenticationHandler());
        }

        public Builder usernamePasswordAuthentication(String username, String password) {
            authenticationHandler = new AnypointUPWAuthenticationHandler(username, password);
            return this;
        }

        public Builder authenticationHandler(@NotNull AuthenticationHandler authenticationHandler) {
            this.authenticationHandler = authenticationHandler;
            return this;
        }

        public Builder proxy(ProxySettings proxySettings) {
            this.proxySettings = proxySettings;
            return this;
        }

        public Builder restClient(RESTClient restClient) {
            this.restClient = restClient;
            return this;
        }

        public Builder anypointRestClient(RESTClientHost anypointRestClient) {
            this.anypointRestClient = anypointRestClient;
            return this;
        }

        public AnypointClient build() {
            ObjectMapper objectMapper = null;
            if (restClient == null) {
                objectMapper = new ObjectMapper();
                objectMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
                objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
                objectMapper.registerModule(new Jdk8Module());
                objectMapper.registerModule(new JavaTimeModule());
                objectMapper.registerModule(new ParameterNamesModule());
                JsonConverterJacksonImpl jsonConverterJackson = new JsonConverterJacksonImpl(objectMapper);
                RESTClient.Builder builder = RESTClient.builder();
                if (proxySettings != null) {
                    builder = builder.proxy(proxySettings);
                }
                final RESTClient.Builder jsonConverter = builder.jsonConverter(jsonConverterJackson);
                restClient = jsonConverter.build();
            }
            if (anypointRestClient == null) {
                final RESTClientHostBuilder hostBuilder = restClient.host(baseUrl);
                if (authenticationHandler != null) {
                    hostBuilder.authenticationHandler(authenticationHandler);
                }
                anypointRestClient = hostBuilder.build();
            }
            final AnypointClient anypointClient = new AnypointClient(baseUrl, restClient, anypointRestClient);
            if (authenticationHandler instanceof AnypointAuthenticationHandler) {
                ((AnypointAuthenticationHandler) authenticationHandler).setAnypointBaseUrl(baseUrl);
            }
            if (objectMapper != null) {
                objectMapper.setInjectableValues(new InjectableValues.Std().addValue("anypointClient", anypointClient));
            }
            return anypointClient;
        }
    }
}
