/*
 * Copyright (c) 2023. Aeontronix Inc
 */

package com.aeontronix.anypointsdk.amc;

import com.aeontronix.anypointsdk.amc.application.AppUpdateStrategy;
import com.aeontronix.anypointsdk.amc.application.ApplicationIdentifier;
import com.aeontronix.anypointsdk.amc.application.deployment.*;
import com.fasterxml.jackson.annotation.JsonIgnore;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

public class AMCAppDeploymentParameters {
    private boolean clustered;
    private String publicUrl;
    private String pathRewrite;
    private boolean lastMileSecurity;
    private boolean forwardSslSession;
    private String runtimeVersion;
    private AppUpdateStrategy updateStrategy = AppUpdateStrategy.ROLLING;
    private boolean disableAmLogForwarding;
    private boolean generateDefaultPublicUrl;
    private int replicas = 1;
    private String desiredState = "STARTED";
    private boolean objectStoreV2Enabled;
    private String vCores = "0.1";
    private Map<String, String> logLevels;
    private String assetOrgId;
    private String assetArtifactId;
    private String assetVersion;

    public AMCAppDeploymentParameters() {
    }

    public AMCAppDeploymentParameters(boolean clustered, String publicUrl, String pathRewrite, boolean lastMileSecurity, boolean forwardSslSession, String runtimeVersion, AppUpdateStrategy updateStrategy, boolean disableAmLogForwarding, boolean generateDefaultPublicUrl, int replicas, String desiredState, boolean objectStoreV2Enabled, String vCores, Map<String, String> logLevels, String assetOrgId, String assetArtifactId, String assetVersion) {
        this.clustered = clustered;
        this.publicUrl = publicUrl;
        this.pathRewrite = pathRewrite;
        this.lastMileSecurity = lastMileSecurity;
        this.forwardSslSession = forwardSslSession;
        this.runtimeVersion = runtimeVersion;
        this.updateStrategy = updateStrategy;
        this.disableAmLogForwarding = disableAmLogForwarding;
        this.generateDefaultPublicUrl = generateDefaultPublicUrl;
        this.replicas = replicas;
        this.desiredState = desiredState;
        this.objectStoreV2Enabled = objectStoreV2Enabled;
        this.vCores = vCores;
        this.logLevels = logLevels;
        this.assetOrgId = assetOrgId;
        this.assetArtifactId = assetArtifactId;
        this.assetVersion = assetVersion;
    }

    public boolean isClustered() {
        return clustered;
    }

    public void setClustered(boolean clustered) {
        this.clustered = clustered;
    }

    public String getPublicUrl() {
        return publicUrl;
    }

    public void setPublicUrl(String publicUrl) {
        this.publicUrl = publicUrl;
    }

    public String getPathRewrite() {
        return pathRewrite;
    }

    public void setPathRewrite(String pathRewrite) {
        this.pathRewrite = pathRewrite;
    }

    public boolean isLastMileSecurity() {
        return lastMileSecurity;
    }

    public void setLastMileSecurity(boolean lastMileSecurity) {
        this.lastMileSecurity = lastMileSecurity;
    }

    public boolean isForwardSslSession() {
        return forwardSslSession;
    }

    public void setForwardSslSession(boolean forwardSslSession) {
        this.forwardSslSession = forwardSslSession;
    }

    public String getRuntimeVersion() {
        return runtimeVersion;
    }

    public void setRuntimeVersion(String runtimeVersion) {
        this.runtimeVersion = runtimeVersion;
    }

    public AppUpdateStrategy getUpdateStrategy() {
        return updateStrategy;
    }

    public void setUpdateStrategy(AppUpdateStrategy updateStrategy) {
        this.updateStrategy = updateStrategy;
    }

    public boolean isDisableAmLogForwarding() {
        return disableAmLogForwarding;
    }

    public void setDisableAmLogForwarding(boolean disableAmLogForwarding) {
        this.disableAmLogForwarding = disableAmLogForwarding;
    }

    public boolean isGenerateDefaultPublicUrl() {
        return generateDefaultPublicUrl;
    }

    public void setGenerateDefaultPublicUrl(boolean generateDefaultPublicUrl) {
        this.generateDefaultPublicUrl = generateDefaultPublicUrl;
    }

    public int getReplicas() {
        return replicas;
    }

    public void setReplicas(int replicas) {
        this.replicas = replicas;
    }

    public String getDesiredState() {
        return desiredState;
    }

    public void setDesiredState(String desiredState) {
        this.desiredState = desiredState;
    }

    public boolean isObjectStoreV2Enabled() {
        return objectStoreV2Enabled;
    }

    public void setObjectStoreV2Enabled(boolean objectStoreV2Enabled) {
        this.objectStoreV2Enabled = objectStoreV2Enabled;
    }

    public String getvCores() {
        return vCores;
    }

    public void setvCores(String vCores) {
        this.vCores = vCores;
    }

    public String getAssetOrgId() {
        return assetOrgId;
    }

    public void setAssetOrgId(String assetOrgId) {
        this.assetOrgId = assetOrgId;
    }

    public String getAssetArtifactId() {
        return assetArtifactId;
    }

    public void setAssetArtifactId(String assetArtifactId) {
        this.assetArtifactId = assetArtifactId;
    }

    public String getAssetVersion() {
        return assetVersion;
    }

    public void setAssetVersion(String assetVersion) {
        this.assetVersion = assetVersion;
    }

    @JsonIgnore
    public synchronized Map<String, String> getLogLevels(boolean create) {
        if (logLevels == null) {
            logLevels = new HashMap<>();
        }
        return logLevels;
    }

    public synchronized Map<String, String> getLogLevels() {
        return logLevels;
    }

    public void setLogLevels(Map<String, String> logLevels) {
        this.logLevels = logLevels;
    }

    public ApplicationDeploymentData toDataObject(String appName,
                                                  String targetId,
                                                  ApplicationIdentifier appId,
                                                  Map<String, String> properties,
                                                  Map<String, String> secureProperties) {
        ApplicationDeploymentData applicationData = new ApplicationDeploymentData();
        applicationData.setName(appName);
        applicationData.setLabels(Collections.singletonList("beta"));
        Target target = new Target();
        target.setProvider("MC");
        target.setTargetId(targetId);
        applicationData.setTarget(target);
        DeploymentSettings deploymentSettings = new DeploymentSettings();
        deploymentSettings.setClustered(clustered);
        deploymentSettings.setEnforceDeployingReplicasAcrossNodes(false);
        deploymentSettings.setJvm(new Jvm());
        deploymentSettings.setRuntimeVersion(runtimeVersion);
        deploymentSettings.setUpdateStrategy(updateStrategy.getJsonValue());
        deploymentSettings.setDisableAmLogForwarding(disableAmLogForwarding);
        deploymentSettings.setPersistentObjectStore(false);
        deploymentSettings.setGenerateDefaultPublicUrl(generateDefaultPublicUrl);
        ApplicationHttp http = new ApplicationHttp();
        Inbound inbound = new Inbound();
        inbound.setPublicUrl(publicUrl);
        inbound.setPathRewrite(pathRewrite);
        inbound.setLastMileSecurity(lastMileSecurity);
        inbound.setForwardSslSession(forwardSslSession);
        http.setInbound(inbound);
        deploymentSettings.setHttp(http);
        target.setDeploymentSettings(deploymentSettings);
        target.setReplicas(replicas);
        ApplicationState applicationConfig = new ApplicationState();
        ApplicationAssetRef ref = new ApplicationAssetRef();
        ref.setGroupId(appId.getGroupId());
        ref.setArtifactId(appId.getArtifactId());
        ref.setVersion(appId.getVersion());
        ref.setPackaging("jar");
        applicationConfig.setRef(ref);
        applicationConfig.setAssets(new ArrayList<>());
        applicationConfig.setDesiredState(desiredState);
        ApplicationServices services = new ApplicationServices();
        ApplicationPropertiesService applicationPropertiesService = new ApplicationPropertiesService();
        applicationPropertiesService.setApplicationName(appName);
        applicationPropertiesService.setProperties(properties);
        applicationPropertiesService.setSecureProperties(secureProperties);
        services.setMuleAgentApplicationPropertiesService(applicationPropertiesService);
        ApplicationLoggingService muleAgentLoggingService = new ApplicationLoggingService();
        if (logLevels != null && !logLevels.isEmpty()) {
            muleAgentLoggingService.setScopeLoggingConfigurations(logLevels.entrySet().stream()
                    .map(e -> new ScopeLoggingConfigurationsItem(e.getKey(), e.getValue()))
                    .collect(Collectors.toList()));
        } else {
            muleAgentLoggingService.setScopeLoggingConfigurations(new ArrayList<>());
        }
        services.setMuleAgentLoggingService(muleAgentLoggingService);
        applicationConfig.setServices(services);
        applicationData.setApplicationState(applicationConfig);
        ApplicationIntegrations integrations = new ApplicationIntegrations();
        integrations.setServices(new Services(new ObjectStoreV2(objectStoreV2Enabled)));
        applicationConfig.setIntegrations(integrations);
        applicationConfig.setvCores(vCores);
        return applicationData;
    }

    @NotNull
    private static HashMap<String, Object> putMap(HashMap<String, Object> map, String key) {
        HashMap<String, Object> childMap = new HashMap<>();
        map.put(key, childMap);
        return childMap;
    }
}
