package com.aeontronix.anypointsdk.exchange;

import com.aeontronix.anypointsdk.AnypointClient;
import com.aeontronix.anypointsdk.apimanager.APIContractData;
import com.aeontronix.commons.exception.UnexpectedException;
import com.aeontronix.restclient.PaginatedResponse;
import com.aeontronix.restclient.RESTClientHostPathBuilder;
import com.aeontronix.restclient.RESTException;
import com.aeontronix.restclient.json.JsonConvertionException;
import org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.stream.Collectors;

public class ExchangeClient {
    private AnypointClient anypointClient;

    public ExchangeClient(AnypointClient anypointClient) {
        this.anypointClient = anypointClient;
    }

    public CreateExchangeAssetRequest createAsset() {
        return new CreateExchangeAssetRequest(anypointClient);
    }

    public CreateExchangeAssetRequest createAsset(String orgId, String assetId, String version, String name) {
        return new CreateExchangeAssetRequest(anypointClient, orgId, assetId, version, name);
    }

    public List<ExchangeClientApplication> listClientApplications(String orgId) throws RESTException {
        try {
            try (PaginatedResponse<ExchangeClientApplicationData> response = anypointClient.getAnypointRestClient().get().path("/exchange/api/v1/organizations").path(orgId)
                    .path("/applications").build().paginate().pageValuesPath("").executePaginated(ExchangeClientApplicationData.class)) {
                return response.toList().stream().map(ExchangeClientApplication::new).collect(Collectors.toList());
            }
        } catch (IOException e) {
            throw new RESTException(e);
        }
    }

    public ExchangeClientApplication findClientApplicationById(String orgId, String id) throws RESTException {
        try {
            return new ExchangeClientApplication(anypointClient.getAnypointRestClient().get().path("/exchange/api/v1/organizations").path(orgId)
                    .path("/applications").path(id).build().executeAndConvertToObject(ExchangeClientApplicationData.class));
        } catch (RESTException e) {
            if (e.getStatusCode() == 403) {
                throw new RESTException("Client application either doesn't exist or not authorized to access", e.getStatusCode(), e.getReason());
            } else {
                throw e;
            }
        }
    }

    public ExchangeClientApplication createClientApplication(@NotNull String masterOrgId, ExchangeClientApplicationData applicationData, String apiInstanceId) throws RESTException {
        try {
            RESTClientHostPathBuilder path = anypointClient.getAnypointRestClient().post()
                    .path("/exchange/api/v2/organizations/").path(masterOrgId).path("/applications");
            if (apiInstanceId != null) {
                path.queryParam("apiInstanceId", apiInstanceId);
            }
            return new ExchangeClientApplication(path.build()
                    .jsonBody(applicationData).executeAndConvertToObject(ExchangeClientApplicationData.class));
        } catch (JsonConvertionException e) {
            throw new UnexpectedException(e);
        }
    }

    public ExchangeClientApplication updateClientApplication(@NotNull String masterOrgId, ExchangeClientApplicationData applicationData) throws RESTException {
        try {
            return new ExchangeClientApplication(anypointClient.getAnypointRestClient().patch()
                    .path("/exchange/api/v2/organizations/").path(masterOrgId).path("/applications")
                    .path(Integer.toString(applicationData.getId())).build()
                    .jsonBody(applicationData).executeAndConvertToObject(ExchangeClientApplicationData.class));
        } catch (JsonConvertionException e) {
            throw new UnexpectedException(e);
        }
    }

    public APIContractData createAPIContract(String applicationId, String apiId, String organizationId, String environmentId, boolean acceptedTerms,
                                             String groupId, String assetId, String version, String productAPIVersion, Long tierId) throws RESTException {
        HashMap<String, Object> req = new HashMap<>();
        req.put("apiId", apiId);
        req.put("environmentId", environmentId);
        req.put("acceptedTerms", acceptedTerms);
        req.put("organizationId", organizationId);
        req.put("groupId", groupId);
        req.put("assetId", assetId);
        req.put("version", version);
        req.put("productAPIVersion", productAPIVersion);
        if (tierId != null) {
            req.put("requestedTierId", tierId);
        }
        try {
            return anypointClient.getAnypointRestClient().post().path("/exchange/api/v1/organizations/").path(organizationId)
                    .path("/applications/").path(applicationId).path("/contracts").build().jsonBody(req)
                    .executeAndConvertToObject(APIContractData.class);
        } catch (JsonConvertionException e) {
            throw new UnexpectedException(e);
        }
    }
}
