/*
 * Copyright (c) 2023. Aeontronix Inc
 */

package com.aeontronix.anypointsdk.export;

import com.aeontronix.anypointsdk.AnypointClient;
import com.aeontronix.anypointsdk.amc.AMCAppDeploymentParameters;
import com.aeontronix.anypointsdk.amc.application.AppUpdateStrategy;
import com.aeontronix.anypointsdk.amc.application.ApplicationSource;
import com.aeontronix.anypointsdk.amc.application.ExchangeApplicationSource;
import com.aeontronix.anypointsdk.amc.application.FileApplicationSource;
import com.aeontronix.anypointsdk.amc.application.deployment.*;
import com.aeontronix.restclient.RESTException;
import org.slf4j.Logger;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import static org.slf4j.LoggerFactory.getLogger;

public class AMCAppExport {
    private static final Logger logger = getLogger(AMCAppExport.class);
    private String file;
    private ApplicationDeploymentData data;

    public AMCAppExport() {
    }

    public AMCAppExport(ApplicationDeploymentData data) {
        this.data = data;
    }

    public ApplicationDeploymentData getData() {
        return data;
    }

    public void setData(ApplicationDeploymentData data) {
        this.data = data;
    }

    public void doImport(ExportFilters filters, AnypointClient client, String organizationId, String envId) throws RESTException, IOException {
        String appName = data.getName();
        Target target = data.getTarget();
        DeploymentSettings deploymentSettings = target.getDeploymentSettings();
        String targetId = target.getTargetId();
        ApplicationState applicationState = data.getApplicationState();
        ApplicationAssetRef ref = applicationState.getRef();
        if (ref == null) {
            ref = new ApplicationAssetRef();
        }
        ApplicationSource applicationSource;
        if (file != null) {
            applicationSource = new FileApplicationSource(new File(file));
        } else {
            applicationSource = new ExchangeApplicationSource(ref.getGroupId(), ref.getArtifactId(), ref.getVersion());
        }
        ApplicationServices appServices = applicationState.getServices(true);
        ApplicationLoggingService muleAgentLoggingService = appServices.getMuleAgentLoggingService(true);
        Set<String> secureProperties = new HashSet<>();
        Map<String, String> properties = new HashMap<>();
        Map<String, String> exportSecureProperty = data.getSecureProperties();
        if (exportSecureProperty != null) {
            for (Map.Entry<String, String> e : exportSecureProperty.entrySet()) {
                secureProperties.add(e.getKey());
                properties.put(e.getKey(), e.getValue());
            }
        }
        Map<String, String> exportNonSecureProperties = data.getNonSecureProperties();
        if (exportNonSecureProperties != null) {
            properties.putAll(exportNonSecureProperties);
        }
        logger.info("Deploying AMC app " + appName);
        Inbound inbound = deploymentSettings.getHttp().getInbound();
        AMCAppDeploymentParameters deploymentParameters = new AMCAppDeploymentParameters(
                deploymentSettings.isClustered(), inbound.getPublicUrl(), inbound.getPathRewrite(),
                inbound.isLastMileSecurity(), inbound.isForwardSslSession(), deploymentSettings.getRuntimeVersion(),
                AppUpdateStrategy.valueOf(deploymentSettings.getUpdateStrategy().toUpperCase()), deploymentSettings.isDisableAmLogForwarding(),
                deploymentSettings.isGenerateDefaultPublicUrl(), target.getReplicas(), applicationState.getDesiredState(),
                true, applicationState.getvCores(), muleAgentLoggingService.getLogLevels(),
                ref.getGroupId(), ref.getArtifactId(), ref.getVersion());
        client.getAMCClient().deployApplication(deploymentParameters, appName, organizationId, envId,
                targetId, applicationSource, properties, secureProperties, null, true);
    }

    public String getFile() {
        return file;
    }

    public void setFile(String file) {
        this.file = file;
    }
}
