/*
 * Copyright (c) 2023. Aeontronix Inc
 */

package com.aeontronix.anypointsdk.export;

import com.aeontronix.anypointsdk.AnypointClient;
import com.aeontronix.anypointsdk.AnypointException;
import com.aeontronix.anypointsdk.amc.AMCClient;
import com.aeontronix.anypointsdk.amc.application.deployment.ApplicationDeploymentData;
import com.aeontronix.anypointsdk.cloudhub.CHApplication;
import com.aeontronix.anypointsdk.cloudhub.CHApplicationData;
import com.aeontronix.anypointsdk.cloudhub.CloudhubClient;
import com.aeontronix.anypointsdk.organization.Environment;
import com.aeontronix.anypointsdk.organization.EnvironmentData;
import com.aeontronix.restclient.RESTException;
import org.slf4j.Logger;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import static org.slf4j.LoggerFactory.getLogger;

public class EnvExport {
    private static final Logger logger = getLogger(EnvExport.class);
    private EnvironmentData data;
    private ArrayList<CloudhubAppExport> cloudhubApps = new ArrayList<>();
    private ArrayList<AMCAppExport> amcApps = new ArrayList<>();

    public EnvExport() {
    }

    public EnvExport(ExportFilters filters, boolean downloadApps, AnypointClient client, Environment environment) throws RESTException, IOException, AnypointException {
        data = environment.getData();
        exportCH1Apps(environment, client, downloadApps);
        exportAMCApps(environment, client);
    }

    private void exportCH1Apps(Environment environment, AnypointClient client, boolean downloadApps) throws AnypointException, RESTException, IOException {
        CloudhubClient cloudhubClient = client.getCloudhubClient();
        for (CHApplication app : cloudhubClient.listCHApplications(environment.getId())) {
            CHApplicationData appData = app.getData();
            CloudhubAppExport appExport = new CloudhubAppExport(appData);
            cloudhubApps.add(appExport);
            String domain = appData.getDomain();
            logger.info("Exporting ch1 app " + domain);
            if (downloadApps) {
                File downloadFile = new File(appExport.getFileName());
                try (FileOutputStream fos = new FileOutputStream(downloadFile)) {
                    logger.info("Downloading application archive " + appData.getFileName());
                    cloudhubClient.downloadApplicationFile(environment.getOrganizationId(), environment.getId(), domain, appData.getFileName(), fos);
                }
            }
        }
    }

    private void exportAMCApps(Environment environment, AnypointClient client) throws RESTException {
        AMCClient amcClient = client.getAMCClient();
        List<ApplicationDeploymentData> apps = amcClient.listDeployments(environment.getOrganizationId(), environment.getId());
        for (ApplicationDeploymentData app : apps) {
            amcApps.add(new AMCAppExport(amcClient.findDeploymentById(environment.getOrganizationId(), environment.getId(),
                    app.getId()).getData()));
        }
    }

    public EnvironmentData getData() {
        return data;
    }

    public void setData(EnvironmentData data) {
        this.data = data;
    }

    public void doImport(ExportFilters filters, AnypointClient client) throws RESTException, IOException {
        for (AMCAppExport amcApp : amcApps) {
            amcApp.doImport(filters, client, data.getOrganizationId(), data.getId());
        }
    }

    public ArrayList<CloudhubAppExport> getCloudhubApps() {
        return cloudhubApps;
    }

    public void setCloudhubApps(ArrayList<CloudhubAppExport> cloudhubApps) {
        this.cloudhubApps = cloudhubApps;
    }

    public ArrayList<AMCAppExport> getAmcApps() {
        return amcApps;
    }

    public void setAmcApps(ArrayList<AMCAppExport> amcApps) {
        this.amcApps = amcApps;
    }
}
