/*
 * Copyright (c) Aeontronix 2021
 */

package com.aeontronix.enhancedmule.config;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.*;
import java.util.HashMap;
import java.util.Map;

public class EMConfig {
    public static final String DEFAULT = "default";
    @JsonProperty("default")
    private String defaultProfileName;
    @JsonProperty("profiles")
    private Map<String, ConfigProfile> profiles;
    private String active;

    public EMConfig() {
    }

    public void save() throws IOException {
        final File file = new File(System.getProperty("user.home") + File.separator + ".enhanced-mule.config.json");
        new ObjectMapper().setSerializationInclusion(JsonInclude.Include.NON_NULL)
                .writerWithDefaultPrettyPrinter().writeValue(file, this);
    }

    public static EMConfig findConfigFile() throws IOException {
        return findConfigFile(null);
    }

    public String getActive() {
        if( active == null ) {
            active = defaultProfileName;
        }
        return active;
    }

    public void setActive(String active) throws ProfileNotFoundException {
        if( active != null ) {
            if( ! profiles.containsKey(active) ) {
                throw new ProfileNotFoundException(active);
            }
            this.active = active;
        } else {
            this.active = defaultProfileName;
        }
    }

    public static EMConfig findConfigFile(String filename) throws IOException {
        if (filename == null) {
            filename = "enhanced-mule.config.json";
        }
        InputStream is = null;
        try {
            is = findConfig(filename);
            if (is != null) {
                return new ObjectMapper().readValue(is, EMConfig.class);
            }
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException e) {
                //
            }
        }
        final EMConfig cfg = new EMConfig();
        cfg.setDefaultProfileName(DEFAULT);
        cfg.getProfiles().put(DEFAULT, new ConfigProfile());
        return cfg;
    }

    private static InputStream findConfig(String filename) throws FileNotFoundException {
        File file = new File(filename);
        if (file.exists()) {
            return new FileInputStream(file);
        }
        file = new File(System.getProperty("user.home") + File.separatorChar + "." + filename);
        if (file.exists()) {
            return new FileInputStream(file);
        }
        file = new File(System.getProperty("user.home") + File.separatorChar + ".enhanced-mule" + File.separatorChar + filename);
        if (file.exists()) {
            return new FileInputStream(file);
        }
        InputStream is = EMConfig.class.getClassLoader().getResourceAsStream(filename);
        if (is != null) {
            return is;
        }
        is = EMConfig.class.getClassLoader().getResourceAsStream("/" + filename);
        if (is != null) {
            return is;
        }
        is = Thread.currentThread().getContextClassLoader().getResourceAsStream(filename);
        if (is != null) {
            return is;
        }
        is = Thread.currentThread().getContextClassLoader().getResourceAsStream("/" + filename);
        return is;
    }

    public ConfigProfile getOrCreateProfile(String profile) {
        ConfigProfile configProfile = profiles.get(profile);
        if( configProfile == null ) {
            configProfile = new ConfigProfile();
        }
        return configProfile;
    }

    public ConfigProfile getProfile(@Nullable String profile, @Nullable String org, @Nullable String groupId) throws ProfileNotFoundException {
        if (profile != null) {
            return getProfileByProfileName(profile);
        }
        ConfigProfile cfgProfile = null;
        if (isNotBlank(org)) {
            cfgProfile = getProfileByOrg(org);
        }
        if (cfgProfile == null && isNotBlank(groupId)) {
            cfgProfile = getProfileByGroupId(groupId);
        }
        if( cfgProfile != null ) {
            return cfgProfile;
        } else {
            return getDefaultProfile();
        }
    }

    public ConfigProfile getProfileByProfileName(@NotNull String profile) throws ProfileNotFoundException {
        ConfigProfile p = profiles.get(profile);
        if (p != null) {
            return p;
        } else if(DEFAULT.equalsIgnoreCase(profile)) {
            p = new ConfigProfile();
            profiles.put(DEFAULT, p);
            return p;
        } else {
            throw new ProfileNotFoundException(profile);
        }
    }

    @Nullable
    public ConfigProfile getProfileByOrg(@NotNull String org) {
        for (ConfigProfile p : profiles.values()) {
            if (p.getOrgs() != null) {
                for (String o : p.getOrgs()) {
                    if (org.equalsIgnoreCase(o)) {
                        return p;
                    }
                }
            }
        }
        return null;
    }

    @Nullable
    public ConfigProfile getProfileByGroupId(@NotNull String groupId) {
        if (isNotBlank(groupId)) {
            for (ConfigProfile p : profiles.values()) {
                if (p.getGroupIds() != null) {
                    for (String g : p.getGroupIds()) {
                        if (g.equalsIgnoreCase(groupId)) {
                            return p;
                        }
                    }
                }
            }
        }
        return null;
    }

    private ConfigProfile getDefaultProfile() {
        final String activeProfileName = getActive();
        if( activeProfileName != null ) {
            return profiles.get(activeProfileName);
        } else {
            return null;
        }
    }

    public String getDefaultProfileName() {
        return defaultProfileName;
    }

    public void setDefaultProfileName(String defaultProfileName) {
        this.defaultProfileName = defaultProfileName;
    }

    @NotNull
    public Map<String, ConfigProfile> getProfiles() {
        if (profiles == null) {
            profiles = new HashMap<>();
        }
        return profiles;
    }

    public void setProfiles(@NotNull Map<String, ConfigProfile> profiles) {
        this.profiles = profiles;
    }

    private static boolean isNotBlank(String str) {
        return str != null && !str.trim().isEmpty();
    }
}
