/*
 * Decompiled with CFR 0.152.
 */
package com.aerospike.client.metrics;

import com.aerospike.client.AerospikeException;
import com.aerospike.client.Host;
import com.aerospike.client.Log;
import com.aerospike.client.async.EventLoop;
import com.aerospike.client.cluster.Cluster;
import com.aerospike.client.cluster.ConnectionStats;
import com.aerospike.client.cluster.Node;
import com.aerospike.client.command.Buffer;
import com.aerospike.client.metrics.Histograms;
import com.aerospike.client.metrics.LatencyBuckets;
import com.aerospike.client.metrics.LatencyType;
import com.aerospike.client.metrics.MetricsListener;
import com.aerospike.client.metrics.MetricsPolicy;
import com.aerospike.client.policy.ClientPolicy;
import com.aerospike.client.util.Util;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public final class MetricsWriter
implements MetricsListener {
    private static final DateTimeFormatter FilenameFormat = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");
    private static final DateTimeFormatter TimestampFormat = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSS");
    private static final int MinFileSize = 1000000;
    private final String dir;
    private final StringBuilder sb;
    private FileWriter writer;
    private long size;
    private long maxSize;
    private int latencyColumns;
    private int latencyShift;
    private boolean enabled;

    public MetricsWriter(String dir) {
        this.dir = dir;
        this.sb = new StringBuilder(8192);
    }

    @Override
    public void onEnable(Cluster cluster, MetricsPolicy policy) {
        if (policy.reportSizeLimit != 0L && policy.reportSizeLimit < 1000000L) {
            throw new AerospikeException("MetricsPolicy.reportSizeLimit " + policy.reportSizeLimit + " must be at least " + 1000000);
        }
        this.maxSize = policy.reportSizeLimit;
        this.latencyColumns = policy.latencyColumns;
        this.latencyShift = policy.latencyShift;
        try {
            Files.createDirectories(Paths.get(this.dir, new String[0]), new FileAttribute[0]);
            this.open();
        }
        catch (IOException ioe) {
            throw new AerospikeException(ioe);
        }
        this.enabled = true;
    }

    @Override
    public void onSnapshot(Cluster cluster) {
        if (this.enabled) {
            this.writeCluster(cluster);
        }
    }

    @Override
    public void onNodeClose(Node node) {
        if (this.enabled) {
            this.sb.setLength(0);
            this.sb.append(LocalDateTime.now().format(TimestampFormat));
            this.sb.append(" node");
            this.writeNode(node);
            this.writeLine();
        }
    }

    @Override
    public void onDisable(Cluster cluster) {
        if (this.enabled) {
            try {
                this.enabled = false;
                this.writeCluster(cluster);
                this.writer.close();
            }
            catch (Throwable e) {
                Log.error("Failed to close metrics writer: " + Util.getErrorMessage(e));
            }
        }
    }

    private void open() throws IOException {
        LocalDateTime now = LocalDateTime.now();
        String path = this.dir + File.separator + "metrics-" + now.format(FilenameFormat) + ".log";
        this.writer = new FileWriter(path, false);
        this.size = 0L;
        this.sb.setLength(0);
        this.sb.append(now.format(TimestampFormat));
        this.sb.append(" header(2)");
        this.sb.append(" cluster[name,clientType,clientVersion,appId,label[],cpu,mem,recoverQueueSize,invalidNodeCount,commandCount,retryCount,delayQueueTimeoutCount,eventloop[],node[]]");
        this.sb.append(" label[name,value]");
        this.sb.append(" eventloop[processSize,queueSize]");
        this.sb.append(" node[name,address,port,syncConn,asyncConn,namespace[]]");
        this.sb.append(" conn[inUse,inPool,opened,closed]");
        this.sb.append(" namespace[name,errors,timeouts,keyBusy,bytesIn,bytesOut,latency[]]");
        this.sb.append(" latency(");
        this.sb.append(this.latencyColumns);
        this.sb.append(',');
        this.sb.append(this.latencyShift);
        this.sb.append(')');
        this.sb.append("[type[l1,l2,l3...]]");
        this.writeLine();
    }

    private void writeCluster(Cluster cluster) {
        Object userBytes;
        MetricsPolicy policy = cluster.getMetricsPolicy();
        ClientPolicy clientPolicy = cluster.client.getClientPolicy();
        String appId = clientPolicy.appId;
        String clusterName = cluster.getClusterName();
        if (clusterName == null) {
            clusterName = "";
        }
        double cpu = Util.getProcessCpuLoad();
        long mem = Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();
        this.sb.setLength(0);
        this.sb.append(LocalDateTime.now().format(TimestampFormat));
        this.sb.append(" cluster[");
        this.sb.append(clusterName);
        this.sb.append(',');
        this.sb.append("java");
        this.sb.append(',');
        this.sb.append(cluster.client.getVersion());
        this.sb.append(',');
        if (appId != null) {
            this.sb.append(appId);
        } else {
            userBytes = cluster.getUser();
            if (userBytes != null && ((byte[])userBytes).length > 0) {
                String user = Buffer.utf8ToString(userBytes, 0, ((byte[])userBytes).length);
                this.sb.append(user);
            }
        }
        this.sb.append(',');
        if (policy.labels != null) {
            this.sb.append("[");
            userBytes = policy.labels.keySet().iterator();
            while (userBytes.hasNext()) {
                String key = (String)userBytes.next();
                this.sb.append("[").append(key).append(",").append(policy.labels.get(key)).append("],");
            }
            this.sb.deleteCharAt(this.sb.length() - 1);
            this.sb.append("]");
        }
        this.sb.append(',');
        this.sb.append((int)cpu);
        this.sb.append(',');
        this.sb.append(mem);
        this.sb.append(',');
        this.sb.append(cluster.getThreadsInUse());
        this.sb.append(',');
        this.sb.append(cluster.getRecoverQueueSize());
        this.sb.append(',');
        this.sb.append(cluster.getInvalidNodeCount());
        this.sb.append(',');
        this.sb.append(cluster.getCommandCount());
        this.sb.append(',');
        this.sb.append(cluster.getRetryCount());
        this.sb.append(',');
        this.sb.append(cluster.getDelayQueueTimeoutCount());
        this.sb.append(",[");
        EventLoop[] eventLoops = cluster.getEventLoopArray();
        if (eventLoops != null) {
            for (int i = 0; i < eventLoops.length; ++i) {
                EventLoop el = eventLoops[i];
                if (i > 0) {
                    this.sb.append(',');
                }
                this.sb.append('[');
                this.sb.append(el.getProcessSize());
                this.sb.append(',');
                this.sb.append(el.getQueueSize());
                this.sb.append(']');
            }
        }
        this.sb.append("],[");
        Node[] nodes = cluster.getNodes();
        for (int i = 0; i < nodes.length; ++i) {
            Node node = nodes[i];
            if (i > 0) {
                this.sb.append(',');
            }
            this.writeNode(node);
        }
        this.sb.append("]");
        this.writeLine();
    }

    private void writeNode(Node node) {
        this.sb.append('[');
        this.sb.append(node.getName());
        this.sb.append(',');
        Host host = node.getHost();
        this.sb.append(host.name);
        this.sb.append(',');
        this.sb.append(host.port);
        this.sb.append(',');
        this.writeConn(node.getConnectionStats());
        this.sb.append(',');
        this.writeConn(node.getAsyncConnectionStats());
        this.sb.append(",[");
        Histograms hGrams = node.getMetrics().getHistograms();
        ConcurrentHashMap<String, LatencyBuckets[]> hMap = hGrams.getMap();
        int max = LatencyType.getMax();
        Iterator<Map.Entry<String, LatencyBuckets[]>> nsItr = hMap.entrySet().iterator();
        while (nsItr.hasNext()) {
            Map.Entry<String, LatencyBuckets[]> entry = nsItr.next();
            String namespace = entry.getKey();
            this.sb.append(namespace).append(',');
            this.sb.append(node.getErrorCountByNS(namespace));
            this.sb.append(',');
            this.sb.append(node.getTimeoutCountbyNS(namespace));
            this.sb.append(',');
            this.sb.append(node.getKeyBusyCountByNS(namespace));
            this.sb.append(',');
            this.sb.append(node.getBytesInByNS(namespace));
            this.sb.append(',');
            this.sb.append(node.getBytesOutByNS(namespace));
            this.sb.append(",[");
            LatencyBuckets[] latencyBuckets = hGrams.getBuckets(namespace);
            for (int i = 0; i < max; ++i) {
                if (i > 0) {
                    this.sb.append(',');
                }
                this.sb.append(LatencyType.getString(i));
                this.sb.append('[');
                LatencyBuckets buckets = latencyBuckets[i];
                int bucketMax = buckets.getMax();
                for (int j = 0; j < bucketMax; ++j) {
                    if (j > 0) {
                        this.sb.append(',');
                    }
                    this.sb.append(buckets.getBucket(j));
                }
                this.sb.append(']');
            }
            if (nsItr.hasNext()) {
                this.sb.append("]],[");
                continue;
            }
            this.sb.append("]]");
        }
        this.sb.append("]]");
    }

    private void writeConn(ConnectionStats cs) {
        this.sb.append(cs.inUse);
        this.sb.append(',');
        this.sb.append(cs.inPool);
        this.sb.append(',');
        this.sb.append(cs.opened);
        this.sb.append(',');
        this.sb.append(cs.closed);
    }

    private void writeLine() {
        try {
            this.sb.append(System.lineSeparator());
            this.writer.write(this.sb.toString());
            this.size += (long)this.sb.length();
            this.writer.flush();
            if (this.maxSize > 0L && this.size >= this.maxSize) {
                this.writer.close();
                this.open();
            }
        }
        catch (IOException ioe) {
            this.enabled = false;
            try {
                this.writer.close();
            }
            catch (Throwable throwable) {
                // empty catch block
            }
            throw new AerospikeException(ioe);
        }
    }
}

