/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.cloud.context;

import com.alibaba.cloud.context.edas.EdasConfiguration;
import com.aliyuncs.AcsRequest;
import com.aliyuncs.AcsResponse;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.http.HttpClientType;
import com.aliyuncs.profile.DefaultProfile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 阿里云SDK，用于调用云产品的POP-API。
 *
 * @author xiaolongzuo
 */
public class AliCloudSdk {

    private static final Logger log = LoggerFactory.getLogger(AliCloudSdk.class);

    private DefaultAcsClient defaultAcsClient;

    private static final Map<String, List<String>> productRegionMap = new HashMap<>();

    static {
        try {
            addEndpoint(AlicloudSdkProduct.EDAS, AlicloudSdkRegion.CN_BEIJING);
            addEndpoint(AlicloudSdkProduct.EDAS, AlicloudSdkRegion.CN_QINGDAO);
            addEndpoint(AlicloudSdkProduct.EDAS, AlicloudSdkRegion.CN_SHENZHEN);
            addEndpoint(AlicloudSdkProduct.EDAS, AlicloudSdkRegion.CN_HANGZHOU);
            addEndpoint(AlicloudSdkProduct.EDAS, AlicloudSdkRegion.CN_SHANGHAI);
            addEndpoint(AlicloudSdkProduct.EDAS, AlicloudSdkRegion.AP_SOUTHEAST_1);
        } catch (ClientException e) {
            log.warn("Edas pop api init failed.", e);
        }
    }

    private static void addEndpoint(AlicloudSdkProduct alicloudSdkProduct, AlicloudSdkRegion alicloudSdkRegion)
        throws ClientException {
        DefaultProfile.addEndpoint(alicloudSdkRegion.getRegionId(), alicloudSdkRegion.getRegionId(),
            alicloudSdkProduct.getProduct(), alicloudSdkRegion.getEndpoint(alicloudSdkProduct));
        List<String> regions = productRegionMap.get(alicloudSdkProduct.getProduct());
        if (regions == null) {
            regions = new ArrayList<>();
            productRegionMap.put(alicloudSdkProduct.getProduct(), regions);
        }
        regions.add(alicloudSdkRegion.getRegionId());
    }

    public AliCloudSdk(AliCloudConfiguration aliCloudConfiguration, String regionId) {
        DefaultProfile defaultProfile = DefaultProfile.getProfile(regionId,
            aliCloudConfiguration.getAccessKey(),
            aliCloudConfiguration.getSecretKey());
        defaultProfile.getHttpClientConfig().setClientType(HttpClientType.Compatible);

        DefaultAcsClient defaultAcsClient = new DefaultAcsClient(defaultProfile);
        defaultAcsClient.setAutoRetry(false);

        this.defaultAcsClient = defaultAcsClient;
    }

    public AliCloudSdk(AliCloudConfiguration aliCloudConfiguration,
                       EdasConfiguration edasConfiguration) {
        this(aliCloudConfiguration, edasConfiguration.getRegionId());
    }

    /**
     * 获取POP-API的响应
     *
     * @param request 请求参数
     * @param <T>     请求泛型
     * @return 响应信息
     * @throws ClientException
     */
    public <T extends AcsResponse> T getAcsResponse(AcsRequest<T> request)
        throws ClientException {
        return defaultAcsClient.getAcsResponse(request);
    }

    public boolean contains(AlicloudSdkProduct alicloudSdkProduct, String region) {
        List<String> regions = productRegionMap.get(alicloudSdkProduct.getProduct());
        return region != null && regions.contains(region);
    }

}
