/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.cloud.context.edas;

import com.alibaba.cloud.context.AliCloudDeploymentMode;
import com.alibaba.cloud.context.utils.AssertUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileReader;
import java.util.Properties;

/**
 * 读取EDAS发布单传递的配置
 *
 * @author xiaolongzuo
 */
class SimpleEdasChangeOrderConfiguration implements EdasChangeOrderConfiguration {

    private static final Logger log = LoggerFactory.getLogger(SimpleEdasChangeOrderConfiguration.class);

    private AliCloudDeploymentMode deploymentMode;

    private String projectName;

    private String tenantId;

    private String addressServerDomain;

    private String spasIdentityFile;

    private String addressServerPort;

    private String configserverClientPort;

    private String eccId;

    private String dauthAccessKey;

    private String dauthSecretKey;

    private String namespace;

    SimpleEdasChangeOrderConfiguration() {
        String deploymentModeString = System.getProperty("alicloud.deployment.mode");
        if (deploymentModeString == null) {
            deploymentModeString = System.getenv("ALICLOUD_DEPLOYMENT_MODE");
        }
        if (deploymentModeString == null) {
            this.deploymentMode = AliCloudDeploymentMode.UNMANAGED;
        } else {
            this.deploymentMode = AliCloudDeploymentMode.valueOf(deploymentModeString);
        }
        if (this.deploymentMode == AliCloudDeploymentMode.EDAS_MANAGED) {
            readEdasChangeOrderConfiguration();
            checkEdasChangeOrderConfiguration();
        }
    }

    private void readEdasChangeOrderConfiguration() {
        this.configserverClientPort = System.getProperty("configserver.client.port");
        this.projectName = System.getProperty("project.name");
        this.addressServerDomain = System.getProperty("address.server.domain");
        this.addressServerPort = System.getProperty("address.server.port");
        this.spasIdentityFile = System.getProperty("spas.identity");
        this.tenantId = System.getProperty("tenant.id");
        this.eccId = System.getProperty("ecc.id");
        this.namespace = System.getProperty("alicloud.edas.namespace");
        try {
            FileReader spasIdentityFileReader = new FileReader(this.spasIdentityFile);
            Properties properties = new Properties();
            properties.load(spasIdentityFileReader);
            this.dauthAccessKey = properties.getProperty("accessKey");
            this.dauthSecretKey = properties.getProperty("secretKey");
        } catch (Exception e) {
            log.error("read spas identity file error.", e);
            throw new RuntimeException(e);
        }
    }

    private void checkEdasChangeOrderConfiguration() {
        AssertUtils.notNull(this.projectName, "Project name can't be null.");
        AssertUtils.notNull(this.tenantId, "Tenant id can't be null.");
        AssertUtils.notNull(this.dauthAccessKey, "Dauth ak can't be null.");
        AssertUtils.notNull(this.dauthSecretKey, "Dauth sk can't be null.");
        AssertUtils.notNull(this.addressServerPort, "Server port can't be null.");
        AssertUtils.notNull(this.addressServerDomain, "Server domain can't be null.");
    }

    @Override
    public boolean isEdasManaged() {
        return this.deploymentMode == AliCloudDeploymentMode.EDAS_MANAGED;
    }

    @Override
    public String getProjectName() {
        return projectName;
    }

    @Override
    public String getTenantId() {
        return tenantId;
    }

    @Override
    public String getAddressServerDomain() {
        return addressServerDomain;
    }

    @Override
    public String getSpasIdentityFile() {
        return spasIdentityFile;
    }

    @Override
    public String getAddressServerPort() {
        return addressServerPort;
    }

    @Override
    public String getConfigserverClientPort() {
        return configserverClientPort;
    }

    @Override
    public String getEccId() {
        return eccId;
    }

    @Override
    public String getDauthAccessKey() {
        return dauthAccessKey;
    }

    @Override
    public String getDauthSecretKey() {
        return dauthSecretKey;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    @Override
    public String getRegionId() {
        if (namespace == null) {
            return null;
        }
        return namespace.contains(":") ? namespace.split(":")[0] : namespace;
    }
}
