package com.alibaba.csp.ahas.sentinel.acm;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.alibaba.csp.ahas.sentinel.datasource.parser.ApplicationClusterInfoParser;
import com.alibaba.csp.ahas.sentinel.datasource.parser.ParamFlowRuleParser;
import com.alibaba.csp.ahas.sentinel.datasource.parser.WebFlowRuleParser;
import com.alibaba.csp.ahas.sentinel.datasource.parser.cluster.ClusterAssignStateParser;
import com.alibaba.csp.ahas.sentinel.datasource.parser.cluster.ClusterClientAssignConfigParser;
import com.alibaba.csp.ahas.sentinel.datasource.parser.cluster.ClusterClientCommonConfigParser;
import com.alibaba.csp.ahas.sentinel.datasource.parser.fallback.BlockFallbackConfigListParser;
import com.alibaba.csp.ahas.sentinel.datasource.parser.gateway.GatewayApiDefinitionParser;
import com.alibaba.csp.ahas.sentinel.datasource.parser.gateway.GatewayFlowRuleParser;
import com.alibaba.csp.ahas.sentinel.datasource.parser.setting.SentinelAdapterSettingParser;
import com.alibaba.csp.ahas.sentinel.datasource.parser.setting.SentinelAdaptiveFlowSettingParser;
import com.alibaba.csp.ahas.sentinel.datasource.parser.setting.SentinelGeneralSettingParser;
import com.alibaba.csp.sentinel.adapter.gateway.common.api.ApiDefinition;
import com.alibaba.csp.sentinel.adapter.gateway.common.api.GatewayApiDefinitionManager;
import com.alibaba.csp.sentinel.adapter.gateway.common.rule.GatewayFlowRule;
import com.alibaba.csp.sentinel.adapter.gateway.common.rule.GatewayRuleManager;
import com.alibaba.csp.sentinel.adaptive.config.AdaptiveFlowSettingEntity;
import com.alibaba.csp.sentinel.adaptive.config.AdaptiveFlowSettingManager;
import com.alibaba.csp.sentinel.cluster.ClusterStateManager;
import com.alibaba.csp.sentinel.cluster.client.auth.ClientAuthConfigRegistry;
import com.alibaba.csp.sentinel.cluster.client.auth.ClientCredentialSupplier;
import com.alibaba.csp.sentinel.cluster.client.auth.ssl.CertStorage;
import com.alibaba.csp.sentinel.cluster.client.config.ClusterClientAssignConfig;
import com.alibaba.csp.sentinel.cluster.client.config.ClusterClientConfig;
import com.alibaba.csp.sentinel.cluster.client.config.ClusterClientConfigManager;
import com.alibaba.csp.sentinel.cluster.registry.ConfigSupplierRegistry;
import com.alibaba.csp.sentinel.datasource.acm.DecryptAcmWhiteDataSource;
import com.alibaba.csp.sentinel.datasource.acm.SentinelAcmConstants;
import com.alibaba.csp.sentinel.datasource.acm.parser.DegradeRuleConfigParser;
import com.alibaba.csp.sentinel.datasource.acm.parser.FlowRuleConfigParser;
import com.alibaba.csp.sentinel.datasource.acm.parser.ManualDegradeRuleListParser;
import com.alibaba.csp.sentinel.datasource.acm.parser.RetryRuleListParser;
import com.alibaba.csp.sentinel.datasource.acm.parser.SystemRuleConfigParser;
import com.alibaba.csp.sentinel.log.RecordLog;
import com.alibaba.csp.sentinel.machine.MachineGroupEntity;
import com.alibaba.csp.sentinel.machine.MachineGroupManager;
import com.alibaba.csp.sentinel.retry.RetryRule;
import com.alibaba.csp.sentinel.retry.RetryRuleManager;
import com.alibaba.csp.sentinel.setting.adapter.AdapterSettingManager;
import com.alibaba.csp.sentinel.setting.adapter.SentinelAdapterSettingEntity;
import com.alibaba.csp.sentinel.setting.fallback.BlockFallbackConfig;
import com.alibaba.csp.sentinel.setting.fallback.BlockFallbackConfigManager;
import com.alibaba.csp.sentinel.setting.general.GeneralSettingManager;
import com.alibaba.csp.sentinel.setting.general.SentinelGeneralSettingEntity;
import com.alibaba.csp.sentinel.slots.block.degrade.*;
import com.alibaba.csp.sentinel.slots.block.flow.FlowRule;
import com.alibaba.csp.sentinel.slots.block.flow.FlowRuleManager;
import com.alibaba.csp.sentinel.slots.block.flow.param.ParamFlowRule;
import com.alibaba.csp.sentinel.slots.block.flow.param.ParamFlowRuleManager;
import com.alibaba.csp.sentinel.slots.system.SystemRule;
import com.alibaba.csp.sentinel.slots.system.SystemRuleManager;

import com.alibaba.csp.sentinel.util.function.Supplier;

import com.alibaba.csp.sentinel.web.adapter.common.rule.WebFlowRule;
import com.alibaba.csp.sentinel.web.adapter.common.rule.WebFlowRuleManager;
import com.taobao.csp.ahas.auth.api.AuthUtil;
import com.taobao.csp.ahas.service.api.client.ClientInfoService;
import com.taobao.csp.ahas.service.component.AgwProductCode;
import com.taobao.diamond.exception.DiamondException;

import static com.alibaba.csp.sentinel.datasource.acm.SentinelAcmConstants.SENTINEL_GROUP_ID;

/**
 * @author Eric Zhao
 * @author yunyin
 */
public class DefaultSdkInitService extends AbstractSentinelSdkInitService {

    private final SimpleDecryptor decryptor;

    public DefaultSdkInitService(ClientInfoService clientInfoService) {
        super(clientInfoService);
        this.decryptor = new SimpleDecryptor();
    }

    @Override
    protected void init0(String sdkVersion) throws Exception {
        initClusterFlow(sdkVersion);
        initAcmDataSource();
    }

    /**
     * Init configuration for Sentinel cluster flow control.
     *
     * @param sdkVersion SDK version
     */
    private void initClusterFlow(final String sdkVersion) {
        final String userId = clientInfoService.getUserId();
        final String ahasNamespace = clientInfoService.getNamespace();
        ConfigSupplierRegistry.setNamespaceSupplier(new Supplier<String>() {
            @Override
            public String get() {
                return userId + "|" + ahasNamespace + "|" + clientInfoService.getAppName();
            }
        });
        ClientAuthConfigRegistry.setClientCredentialSupplier(new ClientCredentialSupplier() {
            @Override
            public Map<String, String> getCredentials() {
                Map<String, String> map = new HashMap<String, String>(8);
                map.put("uid", userId);
                map.put("ahasNs", ahasNamespace);
                map.put("sk", AuthUtil.getSecretKey());
                map.put("instanceId", clientInfoService.getAid(AgwProductCode.SENTINEL));
                map.put("clientType", "JAVA_SDK");
                map.put("clientVersion", sdkVersion);
                map.put("cpbId", String.valueOf(1));
                return map;
            }
        });
        CertStorage.createOrUpdateClientCert(clientInfoService.getRegionId(),
            clientInfoService.getAhasEnv());
    }

    /**
     * Init acm data source
     *
     * @throws DiamondException
     */
    private void initAcmDataSource() throws DiamondException {
        String currentEndPoint = clientInfoService.getAcmEndpoint();
        RecordLog.info("[SentinelAcmDataSourceService] ACM endpoint: " + currentEndPoint);
        System.setProperty("ahas.address.server.domain", currentEndPoint);

        String namespace = clientInfoService.getNamespace();
        String userId = clientInfoService.getUserId();
        String consoleTenantId = clientInfoService.getTid();
        String appName = clientInfoService.getAppName();

        initSettingDs(consoleTenantId, userId, namespace, appName);
        initBlockFallbackDs(consoleTenantId, userId, namespace, appName);
        initApplicationClusterInfoDs(consoleTenantId, userId, namespace, appName);
        initClientRuleDs(consoleTenantId, userId, namespace, appName);

        // For legacy embedded mode: we should first register the cluster rule data source supplier
        // then init cluster related data source, or the cluster rules WILL BE LOST!

        initSentinelClusterDs(consoleTenantId, userId, namespace, appName);

        // Register data source related to gateway adapter.
        if (isGatewayAppType()) {
            initApiGatewayDs(consoleTenantId, userId, namespace, appName);
        }
    }

    private void initClientRuleDs(final String tenantId, final String userId, final String namespace,
                                  String appName) throws DiamondException {
        String flowDataId = SentinelAcmConstants.formFlowRuleDataId(userId, namespace, appName);
        FlowRuleManager.register2Property(new DecryptAcmWhiteDataSource<List<FlowRule>>(tenantId, flowDataId,
            SENTINEL_GROUP_ID, new FlowRuleConfigParser(), decryptor).getProperty());

        String degradeDataId = SentinelAcmConstants.formDegradeRuleDataId(userId, namespace, appName);
        DegradeRuleManager.register2Property(
            new DecryptAcmWhiteDataSource<List<DegradeRule>>(tenantId, degradeDataId, SENTINEL_GROUP_ID,
                new DegradeRuleConfigParser(), decryptor).getProperty());

        // Default circuit breaker rule: since 1.9.6
        String defaultCbRuleDataId = SentinelAcmConstants.formDefaultCircuitBreakerRuleDataId(
            userId, namespace, appName);
        DefaultCircuitBreakerRuleManager.register2Property(
            new DecryptAcmWhiteDataSource<List<DegradeRule>>(tenantId, defaultCbRuleDataId, SENTINEL_GROUP_ID,
                new DegradeRuleConfigParser(), decryptor).getProperty());

        String systemDataId = SentinelAcmConstants.formSystemRuleDataId(userId, namespace, appName);
        SystemRuleManager.register2Property(
            new DecryptAcmWhiteDataSource<List<SystemRule>>(tenantId, systemDataId, SENTINEL_GROUP_ID,
                new SystemRuleConfigParser(), decryptor).getProperty());

        // Since AHAS Sentinel client 1.5.0.
        final String paramFlowDataId = SentinelAcmConstants.formParamFlowRuleDataId(userId, namespace, appName);
        ParamFlowRuleManager.register2Property(
            new DecryptAcmWhiteDataSource<List<ParamFlowRule>>(tenantId, paramFlowDataId,
                SENTINEL_GROUP_ID, new ParamFlowRuleParser(), decryptor).getProperty());

        // Retry rule: since 1.8.0
        String retryRuleDataId = SentinelAcmConstants.formRetryRuleDataId(userId, namespace, appName);
        RetryRuleManager.getInstance().register2Property(
            new DecryptAcmWhiteDataSource<List<RetryRule>>(tenantId, retryRuleDataId,
                SENTINEL_GROUP_ID, new RetryRuleListParser(), decryptor).getProperty());

        // Manual degrade rule: since 1.8.4
        String mdrDataId = SentinelAcmConstants.formManualDegradeRuleDataId(userId, namespace, appName);
        ManualDegradeRuleManager.getInstance().register2Property(
            new DecryptAcmWhiteDataSource<List<ManualDegradeRule>>(tenantId, mdrDataId,
                SENTINEL_GROUP_ID, new ManualDegradeRuleListParser(), decryptor).getProperty());

        // Web flow rule: since 1.10.0
        String webFlowRuleDataId = SentinelAcmConstants.formWebFlowRuleDataId(userId, namespace, appName);
        DecryptAcmWhiteDataSource<Set<WebFlowRule>> webFlowRuleDs
                = new DecryptAcmWhiteDataSource<Set<WebFlowRule>>(tenantId, webFlowRuleDataId, SENTINEL_GROUP_ID,
                new WebFlowRuleParser(), decryptor);
        WebFlowRuleManager.register2Property(webFlowRuleDs.getProperty());

        String log = String.format("ACM data-source initialized, flowDataId: %s, degradeDataId: %s, group: %s, tid: %s",
            flowDataId, degradeDataId, SENTINEL_GROUP_ID, tenantId);
        RecordLog.info(log);
    }

    private void initSentinelClusterDs(String tenantId, String userId, String acmNamespace, String appName)
        throws DiamondException {
        String clusterMapDataId = SentinelAcmConstants.formClusterAssignMapDataId(userId, acmNamespace, appName);

        // Init token client related data source.
        DecryptAcmWhiteDataSource<ClusterClientAssignConfig> clientAssignDs
            = new DecryptAcmWhiteDataSource<ClusterClientAssignConfig>(tenantId, clusterMapDataId, SENTINEL_GROUP_ID,
            new ClusterClientAssignConfigParser(), decryptor);
        ClusterClientConfigManager.registerServerAssignProperty(clientAssignDs.getProperty());

        String clientConfigDataId = SentinelAcmConstants.formClusterClientConfigDataId(userId, acmNamespace, appName);
        DecryptAcmWhiteDataSource<ClusterClientConfig> clientConfigDs
            = new DecryptAcmWhiteDataSource<ClusterClientConfig>(tenantId, clientConfigDataId, SENTINEL_GROUP_ID,
            new ClusterClientCommonConfigParser(), decryptor);
        ClusterClientConfigManager.registerClientConfigProperty(clientConfigDs.getProperty());

        // Init cluster state property for extracting mode from cluster map data source.
        DecryptAcmWhiteDataSource<Integer> clusterStateDs = new DecryptAcmWhiteDataSource<Integer>(tenantId,
            clusterMapDataId,
            SENTINEL_GROUP_ID, new ClusterAssignStateParser(), decryptor);
        ClusterStateManager.registerProperty(clusterStateDs.getProperty());
    }

    private void initApiGatewayDs(String tenantId, String userId, String acmNamespace, String appName)
        throws DiamondException {
        RecordLog.info("[SentinelAcmDsService] Initializing data source for API gateway integration");

        String gatewayFlowRuleDataId = SentinelAcmConstants.formGatewayFlowRuleDataId(userId, acmNamespace, appName);

        // Init gateway flow rule data source.
        DecryptAcmWhiteDataSource<Set<GatewayFlowRule>> gatewayFlowRuleDs
            = new DecryptAcmWhiteDataSource<Set<GatewayFlowRule>>(tenantId, gatewayFlowRuleDataId, SENTINEL_GROUP_ID,
            new GatewayFlowRuleParser(), decryptor);
        GatewayRuleManager.register2Property(gatewayFlowRuleDs.getProperty());

        String gatewayApiDefDataId = SentinelAcmConstants.formGatewayApiDefinitionDataId(userId, acmNamespace, appName);

        // Init gateway API definition data source.
        DecryptAcmWhiteDataSource<Set<ApiDefinition>> gatewayApiDefinitionDs
            = new DecryptAcmWhiteDataSource<Set<ApiDefinition>>(tenantId, gatewayApiDefDataId, SENTINEL_GROUP_ID,
            new GatewayApiDefinitionParser(), decryptor);
        GatewayApiDefinitionManager.register2Property(gatewayApiDefinitionDs.getProperty());
    }

    private void initSettingDs(String tenantId, String userId, String acmNamespace, String appName)
        throws DiamondException {
        // We could register the observer for adapter settings here.
        // For example, we could leverage the consumer to update the configuration of Web URL prefix cleaner.

        String gsDataId = SentinelAcmConstants.formGeneralSettingDataId(userId, acmNamespace, appName);
        DecryptAcmWhiteDataSource<SentinelGeneralSettingEntity> generalSettingDs
            = new DecryptAcmWhiteDataSource<SentinelGeneralSettingEntity>(
            tenantId, gsDataId, SENTINEL_GROUP_ID, new SentinelGeneralSettingParser(), decryptor);
        GeneralSettingManager.register2Property(generalSettingDs.getProperty());

        String asDataId = SentinelAcmConstants.formAdapterSettingDataId(userId, acmNamespace, appName);
        DecryptAcmWhiteDataSource<SentinelAdapterSettingEntity> adapterSettingDs
            = new DecryptAcmWhiteDataSource<SentinelAdapterSettingEntity>(
            tenantId, asDataId, SENTINEL_GROUP_ID, new SentinelAdapterSettingParser(), decryptor);
        AdapterSettingManager.register2Property(adapterSettingDs.getProperty());

        String afSettingDataId = SentinelAcmConstants.formAdaptiveFlowSettingDataId(userId,
            acmNamespace, appName);
        DecryptAcmWhiteDataSource<AdaptiveFlowSettingEntity> adaptiveFlowSettingDs
            = new DecryptAcmWhiteDataSource<AdaptiveFlowSettingEntity>(
            tenantId, afSettingDataId, SENTINEL_GROUP_ID, new SentinelAdaptiveFlowSettingParser(), decryptor);
        AdaptiveFlowSettingManager.register2Property(adaptiveFlowSettingDs.getProperty());
    }

    private void initBlockFallbackDs(String tenantId, String userId, String namespace, String appName)
        throws DiamondException {
        String bfcDataId = SentinelAcmConstants.formBlockFallbackConfigDataId(userId, namespace, appName);
        BlockFallbackConfigManager.getInstance().register2Property(
            new DecryptAcmWhiteDataSource<List<BlockFallbackConfig<Object>>>(tenantId, bfcDataId,
                SENTINEL_GROUP_ID, new BlockFallbackConfigListParser(), decryptor).getProperty());
    }

    private void initApplicationClusterInfoDs(String tenantId, String userId, String namespace, String appName)
        throws DiamondException {
        // 应用集群的基本状态信息（如在线机器数）.
        String machineGroupDataId = SentinelAcmConstants.formMachineGroupDataId(userId, namespace, appName);
        MachineGroupManager.register2Property(
            new DecryptAcmWhiteDataSource<MachineGroupEntity>(tenantId, machineGroupDataId,
                SENTINEL_GROUP_ID, new ApplicationClusterInfoParser(), decryptor).getProperty());
    }
}
