/*
 * Copyright 1999-2019 Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.csp.ahas.sentinel.web;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.alibaba.csp.sentinel.Entry;
import com.alibaba.csp.sentinel.EntryType;
import com.alibaba.csp.sentinel.ResourceTypeConstants;
import com.alibaba.csp.sentinel.SphU;
import com.alibaba.csp.sentinel.Tracer;
import com.alibaba.csp.sentinel.adapter.servlet.callback.RequestOriginParser;
import com.alibaba.csp.sentinel.adapter.servlet.callback.UrlCleaner;
import com.alibaba.csp.sentinel.adapter.servlet.callback.WebCallbackManager;
import com.alibaba.csp.sentinel.adapter.servlet.param.HttpServletRequestItemParser;
import com.alibaba.csp.sentinel.adapter.servlet.util.FilterUtil;
import com.alibaba.csp.sentinel.context.ContextUtil;
import com.alibaba.csp.sentinel.slots.block.BlockException;
import com.alibaba.csp.sentinel.slots.block.flow.statuscode.StatusCodeMetricManager;
import com.alibaba.csp.sentinel.util.AssertUtil;
import com.alibaba.csp.sentinel.util.StringUtil;

import com.alibaba.csp.sentinel.web.adapter.common.param.WebParamParser;

import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.HandlerMapping;
import org.springframework.web.servlet.ModelAndView;

import java.util.Map;

/**
 * @author Eric Zhao
 */
public class SentinelWebInterceptor implements HandlerInterceptor {

    public static final String SENTINEL_ENTRY_ATTR_KEY = "$$sentinel_web_entry";
    public static final String SPRING_WEB_CONTEXT = "sentinel_spring_web_context";

    private final SentinelWebMvcConfig config;

    private final WebParamParser<HttpServletRequest> webParamParser = new WebParamParser<HttpServletRequest>(
        new HttpServletRequestItemParser());

    public SentinelWebInterceptor() {
        this(new SentinelWebMvcConfig());
    }

    public SentinelWebInterceptor(SentinelWebMvcConfig webMvcConfig) {
        AssertUtil.notNull(webMvcConfig, "webMvcConfig cannot be null");
        this.config = webMvcConfig;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler)
        throws Exception {
        Object pattern = request.getAttribute(HandlerMapping.BEST_MATCHING_PATTERN_ATTRIBUTE);
        String apiResourceName;
        if (pattern != null) {
            apiResourceName = (String) pattern;
        } else {
            apiResourceName = FilterUtil.filterTarget(request);
        }

        UrlCleaner urlCleaner = WebCallbackManager.getUrlCleaner();
        if (urlCleaner != null) {
            apiResourceName = urlCleaner.clean(apiResourceName);
        }
        if (StringUtil.isEmpty(apiResourceName)) {
            return true;
        }
        // Add method specification if necessary
        if (config.isHttpMethodSpecify()) {
            apiResourceName = request.getMethod().toUpperCase() + ":" + apiResourceName;
        }

        // Parse the request origin using registered origin parser.
        String origin = parseOrigin(request);
        ContextUtil.enter(SPRING_WEB_CONTEXT, origin);

        Entry curEntry = getCurrentEntry(request);
        if (curEntry != null) {
            return true;
        }

        Map<String, Object> params = webParamParser.parseParameterFor(apiResourceName, request, null);

        try {
            // Note that AsyncEntry is REQUIRED here (for async Servlet).
            // TODO: identify whether request is actually ASYNC here.
            Entry entry = SphU.asyncEntry(apiResourceName, ResourceTypeConstants.COMMON_WEB, EntryType.IN, 1, params);
            request.setAttribute(SENTINEL_ENTRY_ATTR_KEY, entry);
            return true;
        } catch (BlockException ex) {
            try {
                // Return the block page, or redirect to another URL.
                WebCallbackManager.getUrlBlockHandler().blocked(request, response, apiResourceName, ex);

                // Record HTTP status code on request blocked.
                int statusCode = response.getStatus();
                StatusCodeMetricManager.getInstance().recordStatusCode(apiResourceName, statusCode);
            } finally {
                ContextUtil.exit();
            }
            return false;
        }
    }

    private Entry getCurrentEntry(HttpServletRequest request) {
        Object e = request.getAttribute(SENTINEL_ENTRY_ATTR_KEY);
        if (e == null) {
            return null;
        }
        return (Entry) e;
    }

    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler,
                           ModelAndView modelAndView) throws Exception {

    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex)
        throws Exception {
        Entry entry = getCurrentEntry(request);
        if (entry == null) {
            return;
        }
        if (ex != null) {
            Tracer.traceEntry(ex, entry);
        }
        // Record HTTP status code.
        int statusCode = response.getStatus();
        StatusCodeMetricManager.getInstance().recordStatusCode(entry.getResourceWrapper().getName(), statusCode);

        entry.exit();
        ContextUtil.exit();
        request.removeAttribute(SENTINEL_ENTRY_ATTR_KEY);
    }

    private String parseOrigin(HttpServletRequest request) {
        RequestOriginParser originParser = WebCallbackManager.getRequestOriginParser();
        String origin = EMPTY_ORIGIN;
        if (originParser != null) {
            origin = originParser.parseOrigin(request);
            if (StringUtil.isEmpty(origin)) {
                return EMPTY_ORIGIN;
            }
        }
        return origin;
    }

    private static final String EMPTY_ORIGIN = "";
}
