/*
 * Decompiled with CFR 0.152.
 */
package com.alibaba.fluss.fs;

import com.alibaba.fluss.fs.FSDataInputStream;
import com.alibaba.fluss.fs.FSDataOutputStream;
import com.alibaba.fluss.fs.FileSystem;
import com.alibaba.fluss.fs.FsPath;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.UUID;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public abstract class FileSystemBehaviorTestSuite {
    protected FileSystem fs;
    private FsPath basePath;

    protected abstract FileSystem getFileSystem() throws Exception;

    protected abstract FsPath getBasePath() throws Exception;

    protected long getConsistencyToleranceNS() {
        return 0L;
    }

    @BeforeEach
    void prepare() throws Exception {
        this.fs = this.getFileSystem();
        this.basePath = new FsPath(this.getBasePath(), FileSystemBehaviorTestSuite.randomName());
        this.fs.mkdirs(this.basePath);
    }

    @AfterEach
    void cleanup() throws Exception {
        this.fs.delete(this.basePath, true);
    }

    @Test
    void testPathAndScheme() throws Exception {
        Assertions.assertThat((URI)this.fs.getUri()).isEqualTo((Object)this.getBasePath().getFileSystem().getUri());
        Assertions.assertThat((String)this.fs.getUri().getScheme()).isEqualTo(this.getBasePath().toUri().getScheme());
    }

    @Test
    void testFileExists() throws IOException {
        FsPath filePath = this.createRandomFileInDirectory(this.basePath);
        Assertions.assertThat((boolean)this.fs.exists(filePath)).isTrue();
    }

    @Test
    void testFileDoesNotExist() throws IOException {
        Assertions.assertThat((boolean)this.fs.exists(new FsPath(this.basePath, FileSystemBehaviorTestSuite.randomName()))).isFalse();
    }

    @Test
    void testExistingFileDeletion() throws IOException {
        this.testSuccessfulDeletion(this.createRandomFileInDirectory(this.basePath), false);
    }

    @Test
    void testExistingFileRecursiveDeletion() throws IOException {
        this.testSuccessfulDeletion(this.createRandomFileInDirectory(this.basePath), true);
    }

    @Test
    void testNotExistingFileDeletion() throws IOException {
        this.testSuccessfulDeletion(new FsPath(this.basePath, FileSystemBehaviorTestSuite.randomName()), false);
    }

    @Test
    void testNotExistingFileRecursiveDeletion() throws IOException {
        this.testSuccessfulDeletion(new FsPath(this.basePath, FileSystemBehaviorTestSuite.randomName()), true);
    }

    @Test
    void testExistingEmptyDirectoryDeletion() throws IOException {
        FsPath path = new FsPath(this.basePath, FileSystemBehaviorTestSuite.randomName());
        this.fs.mkdirs(path);
        this.testSuccessfulDeletion(path, false);
    }

    @Test
    void testExistingEmptyDirectoryRecursiveDeletion() throws IOException {
        FsPath path = new FsPath(this.basePath, FileSystemBehaviorTestSuite.randomName());
        this.fs.mkdirs(path);
        this.testSuccessfulDeletion(path, true);
    }

    private void testSuccessfulDeletion(FsPath path, boolean recursionEnabled) throws IOException {
        this.fs.delete(path, recursionEnabled);
        Assertions.assertThat((boolean)this.fs.exists(path)).isFalse();
    }

    @Test
    void testExistingNonEmptyDirectoryDeletion() throws IOException {
        FsPath directoryPath = new FsPath(this.basePath, FileSystemBehaviorTestSuite.randomName());
        FsPath filePath = this.createRandomFileInDirectory(directoryPath);
        Assertions.assertThatThrownBy(() -> this.fs.delete(directoryPath, false)).isInstanceOf(IOException.class);
        Assertions.assertThat((boolean)this.fs.exists(directoryPath)).isTrue();
        Assertions.assertThat((boolean)this.fs.exists(filePath)).isTrue();
    }

    @Test
    void testExistingNonEmptyDirectoryRecursiveDeletion() throws IOException {
        FsPath directoryPath = new FsPath(this.basePath, FileSystemBehaviorTestSuite.randomName());
        FsPath filePath = this.createRandomFileInDirectory(directoryPath);
        this.fs.delete(directoryPath, true);
        Assertions.assertThat((boolean)this.fs.exists(directoryPath)).isFalse();
        Assertions.assertThat((boolean)this.fs.exists(filePath)).isFalse();
    }

    @Test
    void testExistingNonEmptyDirectoryWithSubDirRecursiveDeletion() throws IOException {
        FsPath level1SubDirWithFile = new FsPath(this.basePath, FileSystemBehaviorTestSuite.randomName());
        FsPath fileInLevel1Subdir = this.createRandomFileInDirectory(level1SubDirWithFile);
        FsPath level2SubDirWithFile = new FsPath(level1SubDirWithFile, FileSystemBehaviorTestSuite.randomName());
        FsPath fileInLevel2Subdir = this.createRandomFileInDirectory(level2SubDirWithFile);
        this.testSuccessfulDeletion(level1SubDirWithFile, true);
        Assertions.assertThat((boolean)this.fs.exists(fileInLevel1Subdir)).isFalse();
        Assertions.assertThat((boolean)this.fs.exists(level2SubDirWithFile)).isFalse();
        Assertions.assertThat((boolean)this.fs.exists(fileInLevel2Subdir)).isFalse();
    }

    @Test
    void testMkdirsReturnsTrueWhenCreatingDirectory() throws Exception {
        FsPath directory = new FsPath(this.basePath, FileSystemBehaviorTestSuite.randomName());
        Assertions.assertThat((boolean)this.fs.mkdirs(directory)).isTrue();
    }

    @Test
    void testMkdirsCreatesParentDirectories() throws Exception {
        FsPath directory = new FsPath(new FsPath(new FsPath(this.basePath, FileSystemBehaviorTestSuite.randomName()), FileSystemBehaviorTestSuite.randomName()), FileSystemBehaviorTestSuite.randomName());
        Assertions.assertThat((boolean)this.fs.mkdirs(directory)).isTrue();
    }

    @Test
    void testMkdirsReturnsTrueForExistingDirectory() throws Exception {
        FsPath directory = new FsPath(this.basePath, FileSystemBehaviorTestSuite.randomName());
        this.createRandomFileInDirectory(directory);
        Assertions.assertThat((int)this.fs.listStatus(directory).length).isGreaterThan(0);
        Assertions.assertThat((boolean)this.fs.mkdirs(directory)).isTrue();
    }

    @Test
    protected void testMkdirsFailsForExistingFile() throws Exception {
        FsPath file = new FsPath(this.getBasePath(), FileSystemBehaviorTestSuite.randomName());
        this.createFile(file);
        try {
            this.fs.mkdirs(file);
            Assertions.fail((String)"should fail with an IOException");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    void testMkdirsFailsWithExistingParentFile() throws Exception {
        FsPath file = new FsPath(this.getBasePath(), FileSystemBehaviorTestSuite.randomName());
        this.createFile(file);
        FsPath dirUnderFile = new FsPath(file, FileSystemBehaviorTestSuite.randomName());
        try {
            this.fs.mkdirs(dirUnderFile);
            Assertions.fail((String)"should fail with an IOException");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testSimpleFileWriteAndRead() throws Exception {
        String testLine = "Hello Upload!";
        FsPath path = new FsPath(this.basePath, "test.txt");
        try {
            Throwable throwable;
            try (FSDataOutputStream out = this.fs.create(path, FileSystem.WriteMode.OVERWRITE);){
                throwable = null;
                try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)out, StandardCharsets.UTF_8);){
                    writer.write("Hello Upload!");
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
            }
            this.checkPathExistence(path, true, this.getConsistencyToleranceNS());
            var4_4 = null;
            try (FSDataInputStream in = this.fs.open(path);){
                throwable = null;
                try (InputStreamReader ir = new InputStreamReader((InputStream)in, StandardCharsets.UTF_8);
                     BufferedReader reader = new BufferedReader(ir);){
                    String line = reader.readLine();
                    Assertions.assertThat((String)line).isEqualTo("Hello Upload!");
                }
                catch (Throwable throwable3) {
                    throwable = throwable3;
                    throw throwable3;
                }
            }
            catch (Throwable throwable4) {
                var4_4 = throwable4;
                throw throwable4;
            }
        }
        finally {
            this.fs.delete(path, false);
        }
        this.checkPathExistence(path, false, this.getConsistencyToleranceNS());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testDirectoryListing() throws Exception {
        FsPath directory = new FsPath(this.basePath, "testdir/");
        Assertions.assertThat((boolean)this.fs.exists(directory)).isFalse();
        try {
            Assertions.assertThat((boolean)this.fs.mkdirs(directory)).isTrue();
            this.checkEmptyDirectory(directory);
            Assertions.assertThat((Object[])this.fs.listStatus(directory)).isEmpty();
            int numFiles = 3;
            for (int i = 0; i < 3; ++i) {
                FsPath file = new FsPath(directory, "/file-" + i);
                try (FSDataOutputStream out = this.fs.create(file, FileSystem.WriteMode.OVERWRITE);
                     OutputStreamWriter writer = new OutputStreamWriter((OutputStream)out, StandardCharsets.UTF_8);){
                    writer.write("hello-" + i + "\n");
                }
                this.checkPathExistence(file, true, this.getConsistencyToleranceNS());
            }
            Object[] files = this.fs.listStatus(directory);
            Assertions.assertThat((Object[])files).isNotNull();
            Assertions.assertThat((Object[])files).hasSize(3);
            for (Object status : files) {
                Assertions.assertThat((boolean)status.isDir()).isFalse();
            }
            Assertions.assertThat((boolean)this.fs.exists(directory)).isTrue();
            Assertions.assertThat((Object)this.fs.getFileStatus(directory).getPath()).isEqualTo((Object)directory);
        }
        finally {
            FileSystemBehaviorTestSuite.cleanupDirectoryWithRetry(this.fs, directory, this.getConsistencyToleranceNS());
        }
    }

    protected static String randomName() {
        return UUID.randomUUID().toString();
    }

    private void createFile(FsPath file) throws IOException {
        try (FSDataOutputStream out = this.fs.create(file, FileSystem.WriteMode.NO_OVERWRITE);){
            out.write(new byte[]{1, 2, 3, 4, 5, 6, 7, 8});
        }
    }

    private FsPath createRandomFileInDirectory(FsPath directory) throws IOException {
        this.fs.mkdirs(directory);
        FsPath filePath = new FsPath(directory, FileSystemBehaviorTestSuite.randomName());
        this.createFile(filePath);
        return filePath;
    }

    void checkEmptyDirectory(FsPath path) throws IOException, InterruptedException {
        this.checkPathExistence(path, true, this.getConsistencyToleranceNS());
    }

    private void checkPathExistence(FsPath path, boolean expectedExists, long consistencyToleranceNS) throws IOException, InterruptedException {
        if (consistencyToleranceNS == 0L) {
            Assertions.assertThat((boolean)this.fs.exists(path)).isEqualTo(expectedExists);
        } else {
            FileSystemBehaviorTestSuite.checkPathEventualExistence(this.fs, path, expectedExists, consistencyToleranceNS);
        }
    }

    private static void checkPathEventualExistence(FileSystem fs, FsPath path, boolean expectedExists, long consistencyToleranceNS) throws IOException, InterruptedException {
        boolean dirExists;
        long deadline = System.nanoTime() + consistencyToleranceNS;
        while ((dirExists = fs.exists(path)) != expectedExists && System.nanoTime() - deadline < 0L) {
            Thread.sleep(10L);
        }
        Assertions.assertThat((boolean)dirExists).isEqualTo(expectedExists);
    }

    private static void cleanupDirectoryWithRetry(FileSystem fs, FsPath path, long consistencyToleranceNS) throws IOException, InterruptedException {
        fs.delete(path, true);
        long deadline = System.nanoTime() + consistencyToleranceNS;
        while (fs.exists(path) && System.nanoTime() - deadline < 0L) {
            fs.delete(path, true);
            Thread.sleep(50L);
        }
        Assertions.assertThat((boolean)fs.exists(path)).isFalse();
    }
}

