/*
 * Copyright 2017 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package com.alibaba.fluss.shaded.zookeeper3.io.netty.channel.socket.oio;

import com.alibaba.fluss.shaded.zookeeper3.io.netty.buffer.ByteBufAllocator;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.channel.ChannelException;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.channel.ChannelOption;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.channel.MessageSizeEstimator;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.channel.PreferHeapByteBufAllocator;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.channel.RecvByteBufAllocator;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.channel.WriteBufferWaterMark;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.channel.socket.DatagramChannel;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.channel.socket.DefaultDatagramChannelConfig;

import java.io.IOException;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.Map;

import static com.alibaba.fluss.shaded.zookeeper3.io.netty.channel.ChannelOption.SO_TIMEOUT;

final class DefaultOioDatagramChannelConfig extends DefaultDatagramChannelConfig implements OioDatagramChannelConfig  {

    DefaultOioDatagramChannelConfig(DatagramChannel channel, DatagramSocket javaSocket) {
        super(channel, javaSocket);
        setAllocator(new PreferHeapByteBufAllocator(getAllocator()));
    }

    @Override
    public Map<ChannelOption<?>, Object> getOptions() {
        return getOptions(super.getOptions(), SO_TIMEOUT);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T getOption(ChannelOption<T> option) {
        if (option == SO_TIMEOUT) {
            return (T) Integer.valueOf(getSoTimeout());
        }
        return super.getOption(option);
    }

    @Override
    public <T> boolean setOption(ChannelOption<T> option, T value) {
        validate(option, value);

        if (option == SO_TIMEOUT) {
            setSoTimeout((Integer) value);
        } else {
            return super.setOption(option, value);
        }
        return true;
    }

    @Override
    public OioDatagramChannelConfig setSoTimeout(int timeout) {
        try {
            javaSocket().setSoTimeout(timeout);
        } catch (IOException e) {
            throw new ChannelException(e);
        }
        return this;
    }

    @Override
    public int getSoTimeout() {
        try {
            return javaSocket().getSoTimeout();
        } catch (IOException e) {
            throw new ChannelException(e);
        }
    }

    @Override
    public OioDatagramChannelConfig setBroadcast(boolean broadcast) {
        super.setBroadcast(broadcast);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setInterface(InetAddress interfaceAddress) {
        super.setInterface(interfaceAddress);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setLoopbackModeDisabled(boolean loopbackModeDisabled) {
        super.setLoopbackModeDisabled(loopbackModeDisabled);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setNetworkInterface(NetworkInterface networkInterface) {
        super.setNetworkInterface(networkInterface);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setReuseAddress(boolean reuseAddress) {
        super.setReuseAddress(reuseAddress);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setReceiveBufferSize(int receiveBufferSize) {
        super.setReceiveBufferSize(receiveBufferSize);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setSendBufferSize(int sendBufferSize) {
        super.setSendBufferSize(sendBufferSize);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setTimeToLive(int ttl) {
        super.setTimeToLive(ttl);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setTrafficClass(int trafficClass) {
        super.setTrafficClass(trafficClass);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setWriteSpinCount(int writeSpinCount) {
        super.setWriteSpinCount(writeSpinCount);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setConnectTimeoutMillis(int connectTimeoutMillis) {
        super.setConnectTimeoutMillis(connectTimeoutMillis);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setMaxMessagesPerRead(int maxMessagesPerRead) {
        super.setMaxMessagesPerRead(maxMessagesPerRead);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setAllocator(ByteBufAllocator allocator) {
        super.setAllocator(allocator);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setRecvByteBufAllocator(RecvByteBufAllocator allocator) {
        super.setRecvByteBufAllocator(allocator);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setAutoRead(boolean autoRead) {
        super.setAutoRead(autoRead);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setAutoClose(boolean autoClose) {
        super.setAutoClose(autoClose);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setWriteBufferHighWaterMark(int writeBufferHighWaterMark) {
        super.setWriteBufferHighWaterMark(writeBufferHighWaterMark);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setWriteBufferLowWaterMark(int writeBufferLowWaterMark) {
        super.setWriteBufferLowWaterMark(writeBufferLowWaterMark);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setWriteBufferWaterMark(WriteBufferWaterMark writeBufferWaterMark) {
        super.setWriteBufferWaterMark(writeBufferWaterMark);
        return this;
    }

    @Override
    public OioDatagramChannelConfig setMessageSizeEstimator(MessageSizeEstimator estimator) {
        super.setMessageSizeEstimator(estimator);
        return this;
    }
}
