/*
 * Copyright 2015 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package com.alibaba.fluss.shaded.zookeeper3.io.netty.handler.codec.protobuf;

import com.alibaba.fluss.shaded.curator5.com.google.protobuf.ExtensionRegistry;
import com.alibaba.fluss.shaded.curator5.com.google.protobuf.ExtensionRegistryLite;
import com.alibaba.fluss.shaded.curator5.com.google.protobuf.Message;
import com.alibaba.fluss.shaded.curator5.com.google.protobuf.MessageLite;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.buffer.ByteBuf;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.buffer.ByteBufUtil;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.channel.ChannelHandler.Sharable;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.channel.ChannelHandlerContext;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.channel.ChannelPipeline;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.handler.codec.ByteToMessageDecoder;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.handler.codec.LengthFieldBasedFrameDecoder;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.handler.codec.LengthFieldPrepender;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.handler.codec.MessageToMessageDecoder;
import com.alibaba.fluss.shaded.zookeeper3.io.netty.util.internal.ObjectUtil;

import java.util.List;

/**
 * Decodes a received {@link ByteBuf} into a
 * <a href="https://github.com.alibaba.fluss.shaded.curator5.com.google/protobuf">Google Protocol Buffers</a>
 * {@link Message} and {@link MessageLite}. Please note that this decoder must
 * be used with a proper {@link ByteToMessageDecoder} such as {@link ProtobufVarint32FrameDecoder}
 * or {@link LengthFieldBasedFrameDecoder} if you are using a stream-based
 * transport such as TCP/IP. A typical setup for TCP/IP would be:
 * <pre>
 * {@link ChannelPipeline} pipeline = ...;
 *
 * // Decoders
 * pipeline.addLast("frameDecoder",
 *                  new {@link LengthFieldBasedFrameDecoder}(1048576, 0, 4, 0, 4));
 * pipeline.addLast("protobufDecoder",
 *                  new {@link ProtobufDecoder}(MyMessage.getDefaultInstance()));
 *
 * // Encoder
 * pipeline.addLast("frameEncoder", new {@link LengthFieldPrepender}(4));
 * pipeline.addLast("protobufEncoder", new {@link ProtobufEncoder}());
 * </pre>
 * and then you can use a {@code MyMessage} instead of a {@link ByteBuf}
 * as a message:
 * <pre>
 * void channelRead({@link ChannelHandlerContext} ctx, Object msg) {
 *     MyMessage req = (MyMessage) msg;
 *     MyMessage res = MyMessage.newBuilder().setText(
 *                               "Did you say '" + req.getText() + "'?").build();
 *     ch.write(res);
 * }
 * </pre>
 */
@Sharable
public class ProtobufDecoder extends MessageToMessageDecoder<ByteBuf> {

    private static final boolean HAS_PARSER;

    static {
        boolean hasParser = false;
        try {
            // MessageLite.getParserForType() is not available until protobuf 2.5.0.
            MessageLite.class.getDeclaredMethod("getParserForType");
            hasParser = true;
        } catch (Throwable t) {
            // Ignore
        }

        HAS_PARSER = hasParser;
    }

    private final MessageLite prototype;
    private final ExtensionRegistryLite extensionRegistry;

    /**
     * Creates a new instance.
     */
    public ProtobufDecoder(MessageLite prototype) {
        this(prototype, null);
    }

    public ProtobufDecoder(MessageLite prototype, ExtensionRegistry extensionRegistry) {
        this(prototype, (ExtensionRegistryLite) extensionRegistry);
    }

    public ProtobufDecoder(MessageLite prototype, ExtensionRegistryLite extensionRegistry) {
        this.prototype = ObjectUtil.checkNotNull(prototype, "prototype").getDefaultInstanceForType();
        this.extensionRegistry = extensionRegistry;
    }

    @Override
    protected void decode(ChannelHandlerContext ctx, ByteBuf msg, List<Object> out)
            throws Exception {
        final byte[] array;
        final int offset;
        final int length = msg.readableBytes();
        if (msg.hasArray()) {
            array = msg.array();
            offset = msg.arrayOffset() + msg.readerIndex();
        } else {
            array = ByteBufUtil.getBytes(msg, msg.readerIndex(), length, false);
            offset = 0;
        }

        if (extensionRegistry == null) {
            if (HAS_PARSER) {
                out.add(prototype.getParserForType().parseFrom(array, offset, length));
            } else {
                out.add(prototype.newBuilderForType().mergeFrom(array, offset, length).build());
            }
        } else {
            if (HAS_PARSER) {
                out.add(prototype.getParserForType().parseFrom(
                        array, offset, length, extensionRegistry));
            } else {
                out.add(prototype.newBuilderForType().mergeFrom(
                        array, offset, length, extensionRegistry).build());
            }
        }
    }
}
