package com.aliyun.httpcomponent.httpclient;

import com.aliyun.core.http.*;
import com.aliyun.core.logging.ClientLogger;
import com.aliyun.core.utils.BinaryUtils;
import com.aliyun.core.utils.Context;
import com.aliyun.core.utils.StringUtils;
import com.aliyun.httpcomponent.httpclient.implementation.ApacheAsyncHttpResponse;
import com.aliyun.httpcomponent.httpclient.implementation.StreamRequestProducer;
import com.aliyun.httpcomponent.httpclient.implementation.reactive.ReactiveApacheHttpResponse;
import com.aliyun.httpcomponent.httpclient.implementation.reactive.ReactiveHttpResponse;
import com.aliyun.httpcomponent.httpclient.implementation.reactive.ReactiveResponseConsumer;
import org.apache.hc.client5.http.async.methods.SimpleHttpRequest;
import org.apache.hc.client5.http.async.methods.SimpleHttpResponse;
import org.apache.hc.client5.http.async.methods.SimpleRequestBuilder;
import org.apache.hc.client5.http.async.methods.SimpleResponseConsumer;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.async.CloseableHttpAsyncClient;
import org.apache.hc.core5.concurrent.FutureCallback;
import org.apache.hc.core5.http.ContentType;
import org.apache.hc.core5.http.HttpHost;
import org.apache.hc.core5.http.nio.AsyncRequestProducer;
import org.apache.hc.core5.io.CloseMode;
import org.apache.hc.core5.util.Timeout;

import java.net.*;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.regex.Pattern;

class ApacheAsyncHttpClient implements HttpClient {
    private final String RESPONSE_HANDLER_KEY = "RESPONSE_HANDLER";
    private final ClientLogger logger = new ClientLogger(ApacheAsyncHttpClient.class);
    CloseableHttpAsyncClient apacheHttpAsyncClient;
    boolean clientIsStart;

    ApacheAsyncHttpClient(CloseableHttpAsyncClient apacheHttpAsyncClient) {
        this.apacheHttpAsyncClient = apacheHttpAsyncClient;
    }

    @Override
    public CompletableFuture<HttpResponse> send(HttpRequest request) {
        return send(request, Context.NONE);
    }

    @Override
    public CompletableFuture<HttpResponse> send(HttpRequest request, Context context) {
        if (!clientIsStart) {
            clientIsStart = true;
            apacheHttpAsyncClient.start();
        }
        if (request.getStreamBody() != null) {
            if (context.getData(RESPONSE_HANDLER_KEY).isPresent()) {
                return sendV3(request, context);
            }
            return sendV2(request, context);
        } else {
            return sendV1(request, context);
        }
    }

    @Override
    public void close() {
        if (apacheHttpAsyncClient != null)
            apacheHttpAsyncClient.close(CloseMode.GRACEFUL);
    }

    private SimpleHttpRequest toApacheAsyncRequest(HttpRequest request) throws URISyntaxException, ExecutionException, InterruptedException {
        final SimpleRequestBuilder apacheRequestBuilder = SimpleRequestBuilder.create(request.getHttpMethod().toString())
                .setUri(request.getUrl().toURI());
        final HttpHeaders headers = request.getHeaders();
        for (HttpHeader httpHeader : headers) {
            apacheRequestBuilder.setHeader(httpHeader.getName(), httpHeader.getValue());
        }

        switch (request.getHttpMethod()) {
            case GET:
            case HEAD:
            case DELETE:
                return apacheRequestBuilder.build();
            default:
                ContentType type;
                if (StringUtils.isEmpty(headers.getValue("content-type"))) {
                    type = ContentType.APPLICATION_FORM_URLENCODED;
                } else {
                    type = ContentType.create(headers.getValue("content-type"));
                }
                if (request.getBody() != null) {
                    apacheRequestBuilder.setBody(BinaryUtils.copyAllBytesFrom(request.getBody().get()), type);
                }
                return apacheRequestBuilder.build();
        }
    }

    private CompletableFuture<HttpResponse> sendV1(HttpRequest request, Context context) {
        Objects.requireNonNull(request.getHttpMethod(), "'request.getHttpMethod()' cannot be null.");
        Objects.requireNonNull(request.getUrl(), "'request.getUrl()' cannot be null.");
        Objects.requireNonNull(request.getUrl().getProtocol(), "'request.getUrl().getProtocol()' cannot be null.");

        SimpleHttpRequest apacheRequest;
        try {
            apacheRequest = toApacheAsyncRequest(request);
        } catch (URISyntaxException | ExecutionException | InterruptedException e) {
            throw logger.logExceptionAsWarning(new IllegalArgumentException("'url' must can convert to a valid URI", e));
        }

        // set individual request config
        apacheRequest.setConfig(new ApacheIndividualRequestBuilder(request).build());
//        ProxyOptions proxyOptions = request.getProxyOptions();
//        if (proxyOptions != null && proxyOptions.getUsername() != null) {
//            String userInfo = proxyOptions.getUsername() + ":" + proxyOptions.getPassword();
//            apacheRequest.setAuthority(new URIAuthority(
//                    userInfo,
//                    proxyOptions.getAddress().getHostString(),
//                    proxyOptions.getAddress().getPort()));
//        }
        CompletableFuture<SimpleHttpResponse> cf = new CompletableFuture<>();
        final Future<SimpleHttpResponse> future = apacheHttpAsyncClient.execute(
                apacheRequest,
                new FutureCallback<SimpleHttpResponse>() {
                    @Override
                    public void completed(SimpleHttpResponse response) {
                        cf.complete(response);
                    }

                    @Override
                    public void failed(final Exception ex) {
                        cf.completeExceptionally(ex);
                    }

                    @Override
                    public void cancelled() {
                        cf.cancel(true);
                    }
                });
        return cf.thenApply(simpleHttpResponse ->
                new ApacheAsyncHttpResponse(request, simpleHttpResponse));
    }

    private AsyncRequestProducer toApacheRequestProducer(HttpRequest request) throws URISyntaxException {
        final SimpleRequestBuilder apacheRequestBuilder = SimpleRequestBuilder.create(request.getHttpMethod().toString())
                .setUri(request.getUrl().toURI());
        final HttpHeaders headers = request.getHeaders();
        for (HttpHeader httpHeader : headers) {
            apacheRequestBuilder.setHeader(httpHeader.getName(), httpHeader.getValue());
        }
        SimpleHttpRequest simpleHttpRequest = apacheRequestBuilder.build();
        return StreamRequestProducer.create(simpleHttpRequest, request.getStreamBody());
    }

    private CompletableFuture<HttpResponse> sendV2(HttpRequest request, Context context) {
        Objects.requireNonNull(request.getHttpMethod(), "'request.getHttpMethod()' cannot be null.");
        Objects.requireNonNull(request.getUrl(), "'request.getUrl()' cannot be null.");
        Objects.requireNonNull(request.getUrl().getProtocol(), "'request.getUrl().getProtocol()' cannot be null.");

        AsyncRequestProducer apacheRequestProducer;
        try {
            apacheRequestProducer = toApacheRequestProducer(request);
        } catch (URISyntaxException e) {
            throw logger.logExceptionAsWarning(new IllegalArgumentException("'url' must can convert to a valid URI", e));
        }

        CompletableFuture<SimpleHttpResponse> cf = new CompletableFuture<>();
        final Future<SimpleHttpResponse> future = apacheHttpAsyncClient.execute(
                apacheRequestProducer,
                SimpleResponseConsumer.create(),
                new FutureCallback<SimpleHttpResponse>() {
                    @Override
                    public void completed(SimpleHttpResponse response) {
                        cf.complete(response);
                    }

                    @Override
                    public void failed(final Exception ex) {
                        cf.completeExceptionally(ex);
                    }

                    @Override
                    public void cancelled() {
                        cf.cancel(true);
                    }
                });
        return cf.thenApply(simpleHttpResponse ->
                new ApacheAsyncHttpResponse(request, simpleHttpResponse));
    }

    private CompletableFuture<HttpResponse> sendV3(HttpRequest request, Context context) {
        Objects.requireNonNull(request.getHttpMethod(), "'request.getHttpMethod()' cannot be null.");
        Objects.requireNonNull(request.getUrl(), "'request.getUrl()' cannot be null.");
        Objects.requireNonNull(request.getUrl().getProtocol(), "'request.getUrl().getProtocol()' cannot be null.");

        AsyncRequestProducer apacheRequestProducer;
        try {
            apacheRequestProducer = toApacheRequestProducer(request);
        } catch (URISyntaxException e) {
            throw logger.logExceptionAsWarning(new IllegalArgumentException("'url' must can convert to a valid URI", e));
        }

        CompletableFuture<ReactiveApacheHttpResponse> cf = new CompletableFuture<>();
        final Future<ReactiveApacheHttpResponse> future = apacheHttpAsyncClient.execute(
                apacheRequestProducer,
                new ReactiveResponseConsumer((HttpResponseHandler) context.getData(RESPONSE_HANDLER_KEY).get()),
                new FutureCallback<ReactiveApacheHttpResponse>() {
                    @Override
                    public void completed(ReactiveApacheHttpResponse response) {
                        cf.complete(response);
                    }

                    @Override
                    public void failed(final Exception ex) {
                        cf.completeExceptionally(ex);
                    }

                    @Override
                    public void cancelled() {
                        cf.cancel(true);
                    }
                });
        return cf.thenApply(reactiveHttpResponse ->
                new ReactiveHttpResponse(request, reactiveHttpResponse));
    }

    private static final class ApacheIndividualRequestBuilder {
        private final ClientLogger logger = new ClientLogger(ApacheIndividualRequestBuilder.class);
        private final HttpRequest request;
        private final RequestConfig.Builder requestConfigBuilder = RequestConfig.custom();
        private final Timeout connectTimeout = Timeout.ofSeconds(10);
        private final Timeout responseTimeout = Timeout.ofSeconds(20);
        private final Timeout connectionRequestTimeout = Timeout.ofSeconds(30);

        public ApacheIndividualRequestBuilder(HttpRequest request) {
            this.request = request;
        }

        ApacheIndividualRequestBuilder setConnectTimeout() {
            if (request.getConnectTimeout() != null)
                requestConfigBuilder.setConnectTimeout(duration2Timeout(request.getConnectTimeout()));
            else
                requestConfigBuilder.setConnectTimeout(connectTimeout);
            return this;
        }

        ApacheIndividualRequestBuilder setResponseTimeout() {
            if (request.getResponseTimeout() != null)
                requestConfigBuilder.setResponseTimeout(duration2Timeout(request.getReadTimeout()));
            else
                requestConfigBuilder.setResponseTimeout(responseTimeout);
            return this;
        }

        ApacheIndividualRequestBuilder setConnectionRequestTimeout() {
            requestConfigBuilder.setConnectionRequestTimeout(connectionRequestTimeout);
            return this;
        }

        // if not set, based on setRoutePlanner config
        ApacheIndividualRequestBuilder setProxy() {
            ProxyOptions proxyOptions = request.getProxyOptions();
            Pattern nonProxyHostsPattern = null;
            if (proxyOptions != null) {
                nonProxyHostsPattern = StringUtils.isEmpty(proxyOptions.getNonProxyHosts())
                        ? null
                        : Pattern.compile(proxyOptions.getNonProxyHosts(), Pattern.CASE_INSENSITIVE);
            }
            if (nonProxyHostsPattern != null && !nonProxyHostsPattern.matcher(request.getUrl().getHost()).matches()) {
                InetSocketAddress inetSocketAddress = proxyOptions.getAddress();
                requestConfigBuilder.setProxy(new HttpHost(
                        proxyOptions.getScheme(),
                        inetSocketAddress.getAddress(),
                        inetSocketAddress.getHostString(),
                        inetSocketAddress.getPort()
                ));
            }
            return this;
        }

        RequestConfig build() {
            this.setConnectTimeout().setResponseTimeout().setConnectionRequestTimeout().setProxy();
            return requestConfigBuilder.build();
        }

        private Timeout duration2Timeout(Duration duration) {
            return Timeout.ofMilliseconds(duration.toMillis());
        }
    }
}