/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package com.aliyun.apache.hc.core5.http.impl.io;

import com.aliyun.apache.hc.core5.annotation.Contract;
import com.aliyun.apache.hc.core5.annotation.ThreadingBehavior;
import com.aliyun.apache.hc.core5.http.ClassicHttpResponse;
import com.aliyun.apache.hc.core5.http.HttpResponseFactory;
import com.aliyun.apache.hc.core5.http.ReasonPhraseCatalog;
import com.aliyun.apache.hc.core5.http.impl.EnglishReasonPhraseCatalog;
import com.aliyun.apache.hc.core5.http.message.BasicClassicHttpResponse;
import com.aliyun.apache.hc.core5.util.Args;

/**
 * Default factory for creating {@link ClassicHttpResponse} objects.
 *
 * @since 4.0
 */
@Contract(threading = ThreadingBehavior.IMMUTABLE_CONDITIONAL)
public class DefaultClassicHttpResponseFactory implements HttpResponseFactory<ClassicHttpResponse> {

    public static final DefaultClassicHttpResponseFactory INSTANCE = new DefaultClassicHttpResponseFactory();

    private final ReasonPhraseCatalog reasonCatalog;

    /**
     * Creates a new response factory with the given catalog.
     *
     * @param catalog   the catalog of reason phrases
     */
    public DefaultClassicHttpResponseFactory(final ReasonPhraseCatalog catalog) {
        this.reasonCatalog = Args.notNull(catalog, "Reason phrase catalog");
    }

    /**
     * Creates a new response factory with the default catalog.
     * The default catalog is {@link EnglishReasonPhraseCatalog}.
     */
    public DefaultClassicHttpResponseFactory() {
        this(EnglishReasonPhraseCatalog.INSTANCE);
    }

    @Override
    public ClassicHttpResponse newHttpResponse(final int status, final String reasonPhrase) {
        return new BasicClassicHttpResponse(status, reasonPhrase);
    }

    @Override
    public ClassicHttpResponse newHttpResponse(final int status) {
        return new BasicClassicHttpResponse(status, this.reasonCatalog, null);
    }

}
