/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package com.aliyun.apache.hc.client5.http.impl.classic;

import java.io.IOException;
import java.util.List;
import java.util.Locale;

import com.aliyun.apache.hc.client5.http.classic.ExecChain;
import com.aliyun.apache.hc.client5.http.classic.ExecChainHandler;
import com.aliyun.apache.hc.client5.http.config.RequestConfig;
import com.aliyun.apache.hc.client5.http.entity.*;
import com.aliyun.apache.hc.client5.http.entity.BrotliDecompressingEntity;
import com.aliyun.apache.hc.client5.http.entity.BrotliInputStreamFactory;
import com.aliyun.apache.hc.client5.http.entity.DecompressingEntity;
import com.aliyun.apache.hc.client5.http.entity.DeflateInputStreamFactory;
import com.aliyun.apache.hc.client5.http.entity.GZIPInputStreamFactory;
import com.aliyun.apache.hc.client5.http.entity.InputStreamFactory;
import com.aliyun.apache.hc.client5.http.protocol.HttpClientContext;
import com.aliyun.apache.hc.core5.annotation.Contract;
import com.aliyun.apache.hc.core5.annotation.Internal;
import com.aliyun.apache.hc.core5.annotation.ThreadingBehavior;
import com.aliyun.apache.hc.core5.http.ClassicHttpRequest;
import com.aliyun.apache.hc.core5.http.ClassicHttpResponse;
import com.aliyun.apache.hc.core5.http.Header;
import com.aliyun.apache.hc.core5.http.HeaderElement;
import com.aliyun.apache.hc.core5.http.HttpEntity;
import com.aliyun.apache.hc.core5.http.HttpException;
import com.aliyun.apache.hc.core5.http.HttpHeaders;
import com.aliyun.apache.hc.core5.http.config.Lookup;
import com.aliyun.apache.hc.core5.http.config.RegistryBuilder;
import com.aliyun.apache.hc.core5.http.message.BasicHeaderValueParser;
import com.aliyun.apache.hc.core5.http.message.MessageSupport;
import com.aliyun.apache.hc.core5.http.message.ParserCursor;
import com.aliyun.apache.hc.core5.util.Args;

/**
 * Request execution handler in the classic request execution chain
 * that is responsible for automatic response content decompression.
 * <p>
 * Further responsibilities such as communication with the opposite
 * endpoint is delegated to the next executor in the request execution
 * chain.
 * </p>
 *
 * @since 5.0
 */
@Contract(threading = ThreadingBehavior.STATELESS)
@Internal
public final class ContentCompressionExec implements ExecChainHandler {

    private final Header acceptEncoding;
    private final Lookup<InputStreamFactory> decoderRegistry;
    private final boolean ignoreUnknown;

    /**
     * An empty immutable {@code String} array.
     */
    private static final String[] EMPTY_STRING_ARRAY = {};

    public ContentCompressionExec(
            final List<String> acceptEncoding,
            final Lookup<InputStreamFactory> decoderRegistry,
            final boolean ignoreUnknown) {

        final boolean brotliSupported = BrotliDecompressingEntity.isAvailable();
        final String[] encoding;
        if (brotliSupported) {
            encoding = new String[] {"gzip", "x-gzip", "deflate", "br"};
        } else {
            encoding = new String[] {"gzip", "x-gzip", "deflate"};
        }
        this.acceptEncoding = MessageSupport.format(HttpHeaders.ACCEPT_ENCODING,
                                                    acceptEncoding != null ? acceptEncoding.toArray(
                                                        EMPTY_STRING_ARRAY) : encoding);

        if (decoderRegistry != null) {
            this.decoderRegistry = decoderRegistry;
        } else {
            final RegistryBuilder<InputStreamFactory> builder = RegistryBuilder.<InputStreamFactory>create()
                .register("gzip", GZIPInputStreamFactory.getInstance())
                .register("x-gzip", GZIPInputStreamFactory.getInstance())
                .register("deflate", DeflateInputStreamFactory.getInstance());
            if (brotliSupported) {
                builder.register("br", BrotliInputStreamFactory.getInstance());
            }
            this.decoderRegistry = builder.build();
        }


        this.ignoreUnknown = ignoreUnknown;
    }

    public ContentCompressionExec(final boolean ignoreUnknown) {
        this(null, null, ignoreUnknown);
    }

    /**
     * Handles {@code gzip} and {@code deflate} compressed entities by using the following
     * decoders:
     * <ul>
     * <li>gzip - see {@link java.util.zip.GZIPInputStream}</li>
     * <li>deflate - see {@link DeflateInputStream}</li>
     * <li>brotli - see {@link org.brotli.dec.BrotliInputStream}</li>
     * </ul>
     */
    public ContentCompressionExec() {
        this(null, null, true);
    }


    @Override
    public ClassicHttpResponse execute(
            final ClassicHttpRequest request,
            final ExecChain.Scope scope,
            final ExecChain chain) throws IOException, HttpException {
        Args.notNull(request, "HTTP request");
        Args.notNull(scope, "Scope");

        final HttpClientContext clientContext = scope.clientContext;
        final RequestConfig requestConfig = clientContext.getRequestConfig();

        /* Signal support for Accept-Encoding transfer encodings. */
        if (!request.containsHeader(HttpHeaders.ACCEPT_ENCODING) && requestConfig.isContentCompressionEnabled()) {
            request.addHeader(acceptEncoding);
        }

        final ClassicHttpResponse response = chain.proceed(request, scope);

        final HttpEntity entity = response.getEntity();
        // entity can be null in case of 304 Not Modified, 204 No Content or similar
        // check for zero length entity.
        if (requestConfig.isContentCompressionEnabled() && entity != null && entity.getContentLength() != 0) {
            final String contentEncoding = entity.getContentEncoding();
            if (contentEncoding != null) {
                final ParserCursor cursor = new ParserCursor(0, contentEncoding.length());
                final HeaderElement[] codecs = BasicHeaderValueParser.INSTANCE.parseElements(contentEncoding, cursor);
                for (final HeaderElement codec : codecs) {
                    final String codecname = codec.getName().toLowerCase(Locale.ROOT);
                    final InputStreamFactory decoderFactory = decoderRegistry.lookup(codecname);
                    if (decoderFactory != null) {
                        response.setEntity(new DecompressingEntity(response.getEntity(), decoderFactory));
                        response.removeHeaders(HttpHeaders.CONTENT_LENGTH);
                        response.removeHeaders(HttpHeaders.CONTENT_ENCODING);
                        response.removeHeaders(HttpHeaders.CONTENT_MD5);
                    } else {
                        if (!"identity".equals(codecname) && !ignoreUnknown) {
                            throw new HttpException("Unsupported Content-Encoding: " + codec.getName());
                        }
                    }
                }
            }
        }
        return response;
    }

}
