package com.aliyun.auth.credentials.provider;

import com.aliyun.auth.credentials.Credential;
import com.aliyun.auth.credentials.ICredential;
import com.aliyun.auth.credentials.exception.*;
import com.aliyun.auth.credentials.http.*;
import com.aliyun.auth.credentials.utils.*;
import com.aliyun.core.utils.StringUtils;
import com.aliyun.core.utils.Validate;
import com.google.gson.Gson;

import java.time.Instant;
import java.util.Map;


public final class RsaKeyPairCredentialProvider extends HttpCredentialProvider {

    /**
     * Default duration for started sessions. Unit of Second
     */
    private int durationSeconds;

    private String regionId;

    /**
     * Unit of millisecond
     */
    private int connectionTimeout;
    private int readTimeout;
    private Credential credential;

    private RsaKeyPairCredentialProvider(BuilderImpl builder) {
        super(builder);
        this.durationSeconds = builder.durationSeconds;
        this.regionId = builder.regionId;
        this.connectionTimeout = builder.connectionTimeout;
        this.readTimeout = builder.readTimeout;
        this.credential = Validate.notNull(builder.credential, "Credentials must not be null.");
        this.buildRefreshCache();
    }

    public static RsaKeyPairCredentialProvider create(Credential credential) {
        return builder().credential(credential).build();
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    @Override
    public RefreshResult<ICredential> refreshCredentials() {
        CompatibleUrlConnClient client = new CompatibleUrlConnClient();
        ParameterHelper parameterHelper = new ParameterHelper();
        HttpRequest httpRequest = new HttpRequest();
        httpRequest.setUrlParameter("Action", "GenerateSessionAccessKey");
        httpRequest.setUrlParameter("Format", "JSON");
        httpRequest.setUrlParameter("Version", "2015-04-01");
        httpRequest.setUrlParameter("DurationSeconds", String.valueOf(durationSeconds));
        httpRequest.setUrlParameter("AccessKeyId", credential.accessKeyId());
        httpRequest.setUrlParameter("RegionId", this.regionId);
        String strToSign = parameterHelper.composeStringToSign(MethodType.GET, httpRequest.getUrlParameters());
        String signature = parameterHelper.signString(strToSign, credential.accessKeySecret() + "&");
        httpRequest.setUrlParameter("Signature", signature);
        httpRequest.setSysMethod(MethodType.GET);
        httpRequest.setSysConnectTimeout(this.connectionTimeout);
        httpRequest.setSysReadTimeout(this.readTimeout);
        httpRequest.setSysUrl(parameterHelper.composeUrl("sts.aliyuncs.com", httpRequest.getUrlParameters(),
                "https"));
        HttpResponse httpResponse = client.syncInvoke(httpRequest);
        Gson gson = new Gson();
        Map<String, Object> map = gson.fromJson(httpResponse.getHttpContentString(), Map.class);
        if(map.containsKey("SessionAccessKey")) {
            Map<String, String> credentials = (Map<String, String>) map.get("SessionAccessKey");
            Instant expiration = ParameterHelper.getUTCDate(credentials.get("Expiration")).toInstant();
            ICredential credential = Credential.builder()
                    .accessKeyId(credentials.get("SessionAccessKeyId"))
                    .accessKeySecret(credentials.get("SessionAccessKeySecret"))
                    .build();
            return RefreshResult.builder(credential)
                    .staleTime(getStaleTime(expiration))
                    .prefetchTime(getPrefetchTime(expiration))
                    .build();
        } else {
            throw new CredentialException(gson.toJson(map));
        }
    }

    public interface Builder extends HttpCredentialProvider.Builder<RsaKeyPairCredentialProvider, Builder> {
        Builder durationSeconds(int durationSeconds);

        Builder regionId(String regionId);

        Builder connectionTimeout(int connectionTimeout);

        Builder readTimeout(int readTimeout);

        Builder credential(Credential credential);

        @Override
        RsaKeyPairCredentialProvider build();
    }

    private static final class BuilderImpl
            extends HttpCredentialProvider.BuilderImpl<RsaKeyPairCredentialProvider, Builder>
            implements Builder {

        private int durationSeconds = 3600;

        private String regionId = "cn-hangzhou";

        /**
         * Unit of millisecond
         */
        private int connectionTimeout = 1000;
        private int readTimeout = 1000;
        private Credential credential;

        public Builder durationSeconds(int durationSeconds) {
            if (!StringUtils.isEmpty(durationSeconds)) {
                this.durationSeconds = durationSeconds;
            }
            return this;
        }

        public Builder regionId(String regionId) {
            if (!StringUtils.isEmpty(regionId)) {
                this.regionId = regionId;
            }
            return this;
        }

        public Builder connectionTimeout(int connectionTimeout) {
            if (!StringUtils.isEmpty(connectionTimeout)) {
                this.connectionTimeout = connectionTimeout;
            }
            return this;
        }

        public Builder readTimeout(int readTimeout) {
            if (!StringUtils.isEmpty(readTimeout)) {
                this.readTimeout = readTimeout;
            }
            return this;
        }

        public Builder credential(Credential credential) {
            this.credential = credential;
            return this;
        }

        @Override
        public RsaKeyPairCredentialProvider build() {
            return new RsaKeyPairCredentialProvider(this);
        }
    }
}
