package com.aliyun.auth.credentials.provider;

import com.aliyun.auth.credentials.Credential;
import com.aliyun.auth.credentials.ICredential;
import com.aliyun.auth.credentials.exception.*;
import com.aliyun.auth.credentials.http.*;
import com.aliyun.auth.credentials.utils.*;
import com.aliyun.core.utils.StringUtils;
import com.aliyun.core.utils.Validate;
import com.google.gson.Gson;

import java.time.Instant;
import java.util.Map;

public final class RamRoleArnCredentialProvider extends HttpCredentialProvider {

    /**
     * Default duration for started sessions. Unit of Second
     */
    private int durationSeconds;
    /**
     * The arn of the role to be assumed.
     */
    private String roleArn;

    /**
     * An identifier for the assumed role session.
     */
    private final String roleSessionName;
    private String regionId;
    private String policy;
    /**
     * Unit of millisecond
     */
    private int connectionTimeout;
    private int readTimeout;
    private Credential credential;

    private RamRoleArnCredentialProvider(BuilderImpl builder) {
        super(builder);
        this.roleSessionName = builder.roleSessionName;
        this.durationSeconds = builder.durationSeconds;
        this.roleArn = builder.roleArn;
        this.regionId = builder.regionId;
        this.policy = builder.policy;
        this.connectionTimeout = builder.connectionTimeout;
        this.readTimeout = builder.readTimeout;
        this.credential = Validate.notNull(builder.credential, "Credentials must not be null.");
        this.buildRefreshCache();
    }

    public static RamRoleArnCredentialProvider create(Credential credential) {
        return builder().credential(credential).build();
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    @Override
    public RefreshResult<ICredential> refreshCredentials() {
        CompatibleUrlConnClient client = new CompatibleUrlConnClient();
        ParameterHelper parameterHelper = new ParameterHelper();
        HttpRequest httpRequest = new HttpRequest();
        httpRequest.setUrlParameter("Action", "AssumeRole");
        httpRequest.setUrlParameter("Format", "JSON");
        httpRequest.setUrlParameter("Version", "2015-04-01");
        httpRequest.setUrlParameter("DurationSeconds", String.valueOf(durationSeconds));
        httpRequest.setUrlParameter("RoleArn", roleArn);
        httpRequest.setUrlParameter("AccessKeyId", credential.accessKeyId());
        httpRequest.setUrlParameter("RegionId", regionId);
        httpRequest.setUrlParameter("RoleSessionName", roleSessionName);
        if (policy != null) {
            httpRequest.setUrlParameter("Policy", policy);
        }
        httpRequest.setSysMethod(MethodType.GET);
        httpRequest.setSysConnectTimeout(connectionTimeout);
        httpRequest.setSysReadTimeout(readTimeout);
        String strToSign = parameterHelper.composeStringToSign(MethodType.GET, httpRequest.getUrlParameters());
        String signature = parameterHelper.signString(strToSign, credential.accessKeySecret() + "&");
        httpRequest.setUrlParameter("Signature", signature);
        httpRequest.setSysUrl(parameterHelper.composeUrl("sts.aliyuncs.com", httpRequest.getUrlParameters(),
                "https"));
        HttpResponse httpResponse = client.syncInvoke(httpRequest);
        Gson gson = new Gson();
        Map<String, Object> map = gson.fromJson(httpResponse.getHttpContentString(), Map.class);
        if (map.containsKey("Credentials")) {
            Map<String, String> credentials = (Map<String, String>) map.get("Credentials");
            Instant expiration = ParameterHelper.getUTCDate(credentials.get("Expiration")).toInstant();
            ICredential credential = Credential.builder()
                    .accessKeyId(credentials.get("AccessKeyId"))
                    .accessKeySecret(credentials.get("AccessKeySecret"))
                    .securityToken(credentials.get("SecurityToken"))
                    .build();
            return RefreshResult.builder(credential)
                    .staleTime(getStaleTime(expiration))
                    .prefetchTime(getPrefetchTime(expiration))
                    .build();
        } else {
            throw new CredentialException(gson.toJson(map));
        }
    }

    public interface Builder extends HttpCredentialProvider.Builder<RamRoleArnCredentialProvider, Builder> {
        Builder roleSessionName(String roleSessionName);

        Builder durationSeconds(int durationSeconds);

        Builder roleArn(String roleArn);

        Builder regionId(String regionId);

        Builder policy(String policy);

        Builder connectionTimeout(int connectionTimeout);

        Builder readTimeout(int readTimeout);

        Builder credential(Credential credential);

        @Override
        RamRoleArnCredentialProvider build();
    }

    private static final class BuilderImpl
            extends HttpCredentialProvider.BuilderImpl<RamRoleArnCredentialProvider, Builder>
            implements Builder {

        private String roleSessionName = "defaultSessionName";
        private int durationSeconds = 3600;
        private String roleArn;
        private String regionId = "cn-hangzhou";
        private String policy;
        private int connectionTimeout = 1000;
        private int readTimeout = 1000;
        private Credential credential;

        public Builder roleSessionName(String roleSessionName) {
            if (!StringUtils.isEmpty(roleSessionName)) {
                this.roleSessionName = roleSessionName;
            }
            return this;
        }

        public Builder durationSeconds(int durationSeconds) {
            if (!StringUtils.isEmpty(durationSeconds)) {
                this.durationSeconds = durationSeconds;
            }
            return this;
        }

        public Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public Builder regionId(String regionId) {
            if (!StringUtils.isEmpty(regionId)) {
                this.regionId = regionId;
            }
            return this;
        }

        public Builder policy(String policy) {
            this.policy = policy;
            return this;
        }

        public Builder connectionTimeout(int connectionTimeout) {
            if (!StringUtils.isEmpty(connectionTimeout)) {
                this.connectionTimeout = connectionTimeout;
            }
            return this;
        }

        public Builder readTimeout(int readTimeout) {
            if (!StringUtils.isEmpty(readTimeout)) {
                this.readTimeout = readTimeout;
            }
            return this;
        }

        public Builder credential(Credential credential) {
            this.credential = credential;
            return this;
        }

        @Override
        public RamRoleArnCredentialProvider build() {
            return new RamRoleArnCredentialProvider(this);
        }
    }
}