package com.aliyun.core.http;

import com.aliyun.core.logging.ClientLogger;

import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;

public class HttpRequest {
    private final ClientLogger logger = new ClientLogger(HttpRequest.class);
    private HttpMethod httpMethod;
    private URL url;
    private HttpHeaders headers;
    private CompletableFuture<ByteBuffer> body;
    private Duration connectTimeout;
    private Duration writeTimeout;
    private Duration responseTimeout;
    private Duration readTimeout;
    private ProxyOptions proxyOptions;
    private InputStream streamBody;

    public HttpRequest(HttpMethod httpMethod) {
        this.httpMethod = httpMethod;
        this.headers = new HttpHeaders();
    }

    public HttpRequest(HttpMethod httpMethod, URL url) {
        this.httpMethod = httpMethod;
        this.url = url;
        this.headers = new HttpHeaders();
    }

    public HttpRequest(HttpMethod httpMethod, String url) {
        this.httpMethod = httpMethod;
        try {
            this.url = new URL(url);
        } catch (MalformedURLException ex) {
            throw logger.logExceptionAsWarning(new IllegalArgumentException("'url' must be a valid URL", ex));
        }
        this.headers = new HttpHeaders();
    }

    public HttpRequest(HttpMethod httpMethod, URL url, HttpHeaders headers, CompletableFuture<ByteBuffer> body) {
        this.httpMethod = httpMethod;
        this.url = url;
        this.headers = headers;
        this.body = body;
    }

    public HttpMethod getHttpMethod() {
        return httpMethod;
    }

    public HttpRequest setHttpMethod(HttpMethod httpMethod) {
        this.httpMethod = httpMethod;
        return this;
    }

    public URL getUrl() {
        return url;
    }

    public HttpRequest setUrl(URL url) {
        this.url = url;
        return this;
    }

    public HttpRequest setUrl(String url) {
        try {
            this.url = new URL(url);
        } catch (MalformedURLException ex) {
            throw logger.logExceptionAsWarning(new IllegalArgumentException("'url' must be a valid URL.", ex));
        }
        return this;
    }

    public HttpHeaders getHeaders() {
        return headers;
    }

    public HttpRequest setHeaders(HttpHeaders headers) {
        this.headers = headers;
        return this;
    }

    public HttpRequest setHeader(String name, String value) {
        headers.set(name, value);
        return this;
    }

    public CompletableFuture<ByteBuffer> getBody() {
        return body;
    }

    public HttpRequest setBody(String content) {
        final byte[] bodyBytes = content.getBytes(StandardCharsets.UTF_8);
        return setBody(bodyBytes);
    }

    public HttpRequest setBody(byte[] content) {
        return setBody(CompletableFuture.completedFuture(ByteBuffer.wrap(content)));
    }

    public HttpRequest setBody(CompletableFuture<ByteBuffer> content) {
        this.body = content;
        return this;
    }

    public HttpRequest setStreamBody(InputStream inputstream) {
        this.streamBody = inputstream;
        return this;
    }

    public InputStream getStreamBody() {
        return this.streamBody;
    }

    public HttpRequest setProxyOptions(ProxyOptions proxyOptions) {
        this.proxyOptions = proxyOptions;
        return this;
    }

    public ProxyOptions getProxyOptions() {
        return this.proxyOptions;
    }

    public Duration getConnectTimeout() {
        return this.connectTimeout;
    }

    public HttpRequest setConnectTimeout(Duration connectTimeout) {
        this.connectTimeout = connectTimeout;
        return this;
    }

    public Duration getWriteTimeout() {
        return this.writeTimeout;
    }

    public HttpRequest setWriteTimeout(Duration writeTimeout) {
        this.writeTimeout = writeTimeout;
        return this;
    }

    public Duration getResponseTimeout() {
        return this.responseTimeout;
    }

    public HttpRequest setResponseTimeout(Duration responseTimeout) {
        this.responseTimeout = responseTimeout;
        return this;
    }

    public Duration getReadTimeout(){
        return this.readTimeout;
    }

    public HttpRequest setReadTimeout(Duration readTimeout) {
        this.readTimeout = readTimeout;
        return this;
    }

    public HttpRequest copy() {
        final HttpHeaders bufferedHeaders = new HttpHeaders(headers);
        return new HttpRequest(httpMethod, url, bufferedHeaders, body);
    }
}
