package com.aliyun.core.http;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class HttpHeaders implements Iterable<HttpHeader> {
    private final Map<String, HttpHeader> headers = new HashMap<>();

    public HttpHeaders() {
    }

    public HttpHeaders(Map<String, String> headers) {
        headers.forEach(this::set);
    }

    public HttpHeaders(Iterable<HttpHeader> headers) {
        for (final HttpHeader header : headers) {
            this.set(header.getName(), header.getValue());
        }
    }

    public int getSize() {
        return headers.size();
    }

    @Deprecated
    public HttpHeaders put(String name, String value) {
        return set(name, value);
    }

    public HttpHeaders set(String name, String value) {
        if (name == null) {
            return this;
        }
        String caseInsensitiveName = formatKey(name);
        if (value == null) {
            remove(caseInsensitiveName);
        } else {
            headers.put(caseInsensitiveName, new HttpHeader(name, value));
        }
        return this;
    }

    public HttpHeaders set(String name, List<String> values) {
        if (name == null) {
            return this;
        }
        String caseInsensitiveName = formatKey(name);
        if (values == null) {
            remove(caseInsensitiveName);
        } else {
            headers.put(caseInsensitiveName, new HttpHeader(name, values));
        }
        return this;
    }

    public HttpHeaders setAll(Map<String, List<String>> headers) {
        headers.forEach(this::set);
        return this;
    }

    public HttpHeader get(String name) {
        return headers.get(formatKey(name));
    }

    public HttpHeader remove(String name) {
        return headers.remove(formatKey(name));
    }

    public String getValue(String name) {
        final HttpHeader header = get(name);
        return header == null ? null : header.getValue();
    }

    public String[] getValues(String name) {
        final HttpHeader header = get(name);
        return header == null ? null : header.getValues();
    }

    private String formatKey(final String key) {
        return key.toLowerCase(Locale.ROOT);
    }

    public Map<String, String> toMap() {
        final Map<String, String> result = new HashMap<>();
        for (final HttpHeader header : headers.values()) {
            result.put(header.getName(), header.getValue());
        }
        return Collections.unmodifiableMap(result);
    }

    public Map<String, String> toCaseInsensitiveMap() {
        final Map<String, String> result = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
        for (final HttpHeader header : headers.values()) {
            result.put(header.getName(), header.getValue());
        }
        return Collections.unmodifiableMap(result);
    }

    public HttpHeaders putAll(HttpHeaders headers) {
        this.headers.putAll(headers.headers);
        return this;
    }

    @Override
    public Iterator<HttpHeader> iterator() {
        return headers.values().iterator();
    }

    public Stream<HttpHeader> stream() {
        return headers.values().stream();
    }

    @Override
    public String toString() {
        return this.stream()
                .map(header -> header.getName() + "=" + header.getValue())
                .collect(Collectors.joining(", "));
    }
}
