package com.aliyun.core.utils;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;

public class Configuration implements Cloneable {

    public static final String PROPERTY_HTTP_PROXY = "HTTP_PROXY";

    public static final String PROPERTY_HTTPS_PROXY = "HTTPS_PROXY";

    public static final String PROPERTY_IDENTITY_ENDPOINT = "IDENTITY_ENDPOINT";

    public static final String PROPERTY_IDENTITY_HEADER = "IDENTITY_HEADER";

    public static final String PROPERTY_NO_PROXY = "NO_PROXY";

    public static final String PROPERTY_MSI_ENDPOINT = "MSI_ENDPOINT";

    public static final String PROPERTY_MSI_SECRET = "MSI_SECRET";

    public static final String PROPERTY_ALIYUN_SUBSCRIPTION_ID = "ALI_SUBSCRIPTION_ID";

    public static final String PROPERTY_ALIYUN_USERNAME = "ALI_USERNAME";

    public static final String PROPERTY_ALIYUN_PASSWORD = "ALI_PASSWORD";


    public static final String PROPERTY_ALIYUN_CLIENT_ID = "ALI_CLIENT_ID";

    public static final String PROPERTY_ALIYUN_CLIENT_SECRET = "ALI_CLIENT_SECRET";

    public static final String PROPERTY_ALIYUN_TENANT_ID = "ALI_TENANT_ID";

    public static final String PROPERTY_ALIYUN_CLIENT_CERTIFICATE_PATH = "ALI_CLIENT_CERTIFICATE_PATH";

    public static final String PROPERTY_ALIYUN_RESOURCE_GROUP = "ALI_RESOURCE_GROUP";

    public static final String PROPERTY_ALIYUN_CLOUD = "ALI_CLOUD";

    public static final String PROPERTY_ALIYUN_AUTHORITY_HOST = "ALI_AUTHORITY_HOST";

    public static final String PROPERTY_ALIYUN_TELEMETRY_DISABLED = "ALI_TELEMETRY_DISABLED";

    public static final String PROPERTY_ALIYUN_LOG_LEVEL = "ALI_SDK_LOG_LEVEL";

    public static final String PROPERTY_ALIYUN_TRACING_DISABLED = "ALI_TRACING_DISABLED";

    private static final String[] DEFAULT_CONFIGURATIONS = {
            PROPERTY_HTTP_PROXY,
            PROPERTY_HTTPS_PROXY,
            PROPERTY_IDENTITY_ENDPOINT,
            PROPERTY_IDENTITY_HEADER,
            PROPERTY_NO_PROXY,
            PROPERTY_MSI_ENDPOINT,
            PROPERTY_MSI_SECRET,
            PROPERTY_ALIYUN_SUBSCRIPTION_ID,
            PROPERTY_ALIYUN_USERNAME,
            PROPERTY_ALIYUN_PASSWORD,
            PROPERTY_ALIYUN_CLIENT_ID,
            PROPERTY_ALIYUN_CLIENT_SECRET,
            PROPERTY_ALIYUN_TENANT_ID,
            PROPERTY_ALIYUN_CLIENT_CERTIFICATE_PATH,
            PROPERTY_ALIYUN_RESOURCE_GROUP,
            PROPERTY_ALIYUN_CLOUD,
            PROPERTY_ALIYUN_AUTHORITY_HOST,
            PROPERTY_ALIYUN_TELEMETRY_DISABLED,
            PROPERTY_ALIYUN_LOG_LEVEL,
            PROPERTY_ALIYUN_TRACING_DISABLED,
    };

    /*
     * Gets the global configuration shared by all client libraries.
     */
    private static final Configuration GLOBAL_CONFIGURATION = new Configuration();

    public static final Configuration NONE = new NoopConfiguration();

    private final ConcurrentMap<String, String> configurations;

    public Configuration() {
        this.configurations = new ConcurrentHashMap<>();
        loadBaseConfiguration(this);
    }

    private Configuration(ConcurrentMap<String, String> configurations) {
        this.configurations = new ConcurrentHashMap<>(configurations);
    }

    public static Configuration getGlobalConfiguration() {
        return GLOBAL_CONFIGURATION;
    }

    public String get(String name) {
        return getOrLoad(name);
    }

    public <T> T get(String name, T defaultValue) {
        return convertOrDefault(getOrLoad(name), defaultValue);
    }

    public <T> T get(String name, Function<String, T> converter) {
        String value = getOrLoad(name);
        if (StringUtils.isEmpty(value)) {
            return null;
        }

        return converter.apply(value);
    }

    private String getOrLoad(String name) {
        String value = configurations.get(name);
        if (value != null) {
            return value;
        }

        value = load(name);
        if (value != null) {
            configurations.put(name, value);
            return value;
        }

        return null;
    }

    private String load(String name) {
        String value = loadFromProperties(name);

        if (value != null) {
            return value;
        }

        return loadFromEnvironment(name);
    }

    String loadFromEnvironment(String name) {
        return System.getenv(name);
    }

    String loadFromProperties(String name) {
        return System.getProperty(name);
    }

    public Configuration put(String name, String value) {
        configurations.put(name, value);
        return this;
    }

    public String remove(String name) {
        return configurations.remove(name);
    }

    public boolean contains(String name) {
        return configurations.containsKey(name);
    }

    @SuppressWarnings("CloneDoesntCallSuperClone")
    public Configuration clone() {
        return new Configuration(configurations);
    }

    @SuppressWarnings("unchecked")
    private static <T> T convertOrDefault(String value, T defaultValue) {
        if (StringUtils.isEmpty(value)) {
            return defaultValue;
        }

        Object convertedValue;
        if (defaultValue instanceof Byte) {
            convertedValue = Byte.parseByte(value);
        } else if (defaultValue instanceof Short) {
            convertedValue = Short.parseShort(value);
        } else if (defaultValue instanceof Integer) {
            convertedValue = Integer.parseInt(value);
        } else if (defaultValue instanceof Long) {
            convertedValue = Long.parseLong(value);
        } else if (defaultValue instanceof Float) {
            convertedValue = Float.parseFloat(value);
        } else if (defaultValue instanceof Double) {
            convertedValue = Double.parseDouble(value);
        } else if (defaultValue instanceof Boolean) {
            convertedValue = Boolean.parseBoolean(value);
        } else {
            convertedValue = value;
        }

        return (T) convertedValue;
    }

    private void loadBaseConfiguration(Configuration configuration) {
        for (String config : DEFAULT_CONFIGURATIONS) {
            String value = load(config);
            if (value != null) {
                configuration.put(config, value);
            }
        }
    }
}
