package com.aliyun.core.http.policy;

import com.aliyun.core.http.*;
import com.aliyun.core.logging.ClientLogger;
import com.aliyun.core.logging.LogLevel;
import com.aliyun.core.utils.StringUtils;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.util.Collections;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

public class HttpLoggingPolicy {
    private static final int MAX_BODY_LOG_SIZE = 1024 * 16;
    private static final String REDACTED_PLACEHOLDER = "REDACTED";
    private final HttpLogDetailLevel httpLogDetailLevel;
    private final Set<String> allowedHeaderNames;
    private final Set<String> allowedQueryParameterNames;
    private final boolean prettyPrintBody;
    public static final String RETRY_COUNT_CONTEXT = "requestRetryCount";

    public HttpLoggingPolicy(HttpLogOptions httpLogOptions) {
        if (httpLogOptions == null) {
            this.httpLogDetailLevel = HttpLogDetailLevel.NONE;
            this.allowedHeaderNames = Collections.emptySet();
            this.allowedQueryParameterNames = Collections.emptySet();
            this.prettyPrintBody = false;
        } else {
            this.httpLogDetailLevel = httpLogOptions.getLogLevel();
            this.allowedHeaderNames = httpLogOptions.getAllowedHeaderNames()
                    .stream()
                    .map(headerName -> headerName.toLowerCase(Locale.ROOT))
                    .collect(Collectors.toSet());
            this.allowedQueryParameterNames = httpLogOptions.getAllowedQueryParamNames()
                    .stream()
                    .map(queryParamName -> queryParamName.toLowerCase(Locale.ROOT))
                    .collect(Collectors.toSet());
            this.prettyPrintBody = httpLogOptions.isPrettyPrintBody();
        }
    }

    private void logRequest(final ClientLogger logger, final HttpRequest request,
                            final Optional<Object> optionalRetryCount) {

        if (!logger.canLogAtLevel(LogLevel.INFORMATIONAL)) {
            return;
        }

        StringBuilder requestLogMessage = new StringBuilder();
        if (httpLogDetailLevel.shouldLogUrl()) {
            requestLogMessage.append("--> ")
                    .append(request.getHttpMethod())
                    .append(" ")
//                    .append(getRedactedUrl(request.getUrl()))
                    .append(System.lineSeparator());

            optionalRetryCount.ifPresent(o -> requestLogMessage.append("Try count: ")
                    .append(o)
                    .append(System.lineSeparator()));
        }

        addHeadersToLogMessage(logger, request.getHeaders(), requestLogMessage);

        if (!httpLogDetailLevel.shouldLogBody()) {
            logAndReturn(logger, requestLogMessage, null);
            return;
        }

        if (request.getBody() == null) {
            requestLogMessage.append("(empty body)")
                    .append(System.lineSeparator())
                    .append("--> END ")
                    .append(request.getHttpMethod())
                    .append(System.lineSeparator());

            logAndReturn(logger, requestLogMessage, null);
            return;
        }

        String contentType = request.getHeaders().getValue("Content-Type");
        long contentLength = getContentLength(logger, request.getHeaders());

        if (shouldBodyBeLogged(contentType, contentLength)) {
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream((int) contentLength);
            requestLogMessage.append(contentLength)
                    .append("-byte body:")
                    .append(System.lineSeparator())
                    .append(convertStreamToString(outputStream, logger))
                    .append(System.lineSeparator())
                    .append("--> END ")
                    .append(request.getHttpMethod())
                    .append(System.lineSeparator());

            logger.info(requestLogMessage.toString());
        } else {
            requestLogMessage.append(contentLength)
                    .append("-byte body: (content not logged)")
                    .append(System.lineSeparator())
                    .append("--> END ")
                    .append(request.getHttpMethod())
                    .append(System.lineSeparator());

            logAndReturn(logger, requestLogMessage, null);
        }
    }

    /*
     * Logs thr HTTP response.
     */
    private HttpResponse logResponse(final ClientLogger logger, final HttpResponse response, long startNs) throws IOException {
        if (!logger.canLogAtLevel(LogLevel.INFORMATIONAL)) {
            return response;
        }

        long tookMs = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNs);

        String contentLengthString = response.getHeaderValue("Content-Length");
        String bodySize = (StringUtils.isEmpty(contentLengthString))
                ? "unknown-length body"
                : contentLengthString + "-byte body";

        StringBuilder responseLogMessage = new StringBuilder();
        if (httpLogDetailLevel.shouldLogUrl()) {
            responseLogMessage.append("<-- ")
                    .append(response.getStatusCode())
                    .append(" ")
//                    .append(getRedactedUrl(response.getRequest().getUrl()))
                    .append(" (")
                    .append(tookMs)
                    .append(" ms, ")
                    .append(bodySize)
                    .append(")")
                    .append(System.lineSeparator());
        }

        addHeadersToLogMessage(logger, response.getHeaders(), responseLogMessage);

        if (!httpLogDetailLevel.shouldLogBody()) {
            responseLogMessage.append("<-- END HTTP");
            return logAndReturn(logger, responseLogMessage, response);
        }

        String contentTypeHeader = response.getHeaderValue("Content-Type");
        long contentLength = getContentLength(logger, response.getHeaders());

        if (shouldBodyBeLogged(contentTypeHeader, contentLength)) {
            HttpResponse bufferedResponse = response.buffer();
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream((int) contentLength);
            WritableByteChannel bodyContentChannel = Channels.newChannel(outputStream);
            writeBufferToBodyStream(bodyContentChannel, bufferedResponse.getBody());
            responseLogMessage.append("Response body:")
                    .append(System.lineSeparator())
                    .append(convertStreamToString(outputStream, logger))
                    .append(System.lineSeparator())
                    .append("<-- END HTTP");
            logger.info(responseLogMessage.toString());
            return bufferedResponse;
        } else {
            responseLogMessage.append("(body content not logged)")
                    .append(System.lineSeparator())
                    .append("<-- END HTTP");

            return logAndReturn(logger, responseLogMessage, response);
        }
    }

    private <T> T logAndReturn(ClientLogger logger, StringBuilder logMessageBuilder, T data) {
        logger.info(logMessageBuilder.toString());
        return data;
    }

    private String getAllowedQueryString(String queryString) {
        if (StringUtils.isEmpty(queryString)) {
            return "";
        }

        StringBuilder queryStringBuilder = new StringBuilder();
        String[] queryParams = queryString.split("&");
        for (String queryParam : queryParams) {
            if (queryStringBuilder.length() > 0) {
                queryStringBuilder.append("&");
            }

            String[] queryPair = queryParam.split("=", 2);
            if (queryPair.length == 2) {
                String queryName = queryPair[0];
                if (allowedQueryParameterNames.contains(queryName.toLowerCase(Locale.ROOT))) {
                    queryStringBuilder.append(queryParam);
                } else {
                    queryStringBuilder.append(queryPair[0]).append("=").append(REDACTED_PLACEHOLDER);
                }
            } else {
                queryStringBuilder.append(queryParam);
            }
        }

        return queryStringBuilder.toString();
    }

    /*
     * Adds HTTP headers into the StringBuilder that is generating the log message.
     *
     * @param headers HTTP headers on the request or response.
     * @param sb StringBuilder that is generating the log message.
     * @param logLevel Log level the environment is configured to use.
     */
    private void addHeadersToLogMessage(ClientLogger logger, HttpHeaders headers, StringBuilder sb) {
        // Either headers shouldn't be logged or the logging level isn't set to VERBOSE, don't add headers.
        if (!httpLogDetailLevel.shouldLogHeaders() || !logger.canLogAtLevel(LogLevel.VERBOSE)) {
            return;
        }

        for (HttpHeader header : headers) {
            String headerName = header.getName();
            sb.append(headerName).append(":");
            if (allowedHeaderNames.contains(headerName.toLowerCase(Locale.ROOT))) {
                sb.append(header.getValue());
            } else {
                sb.append(REDACTED_PLACEHOLDER);
            }
            sb.append(System.lineSeparator());
        }
    }

    private long getContentLength(ClientLogger logger, HttpHeaders headers) {
        long contentLength = 0;

        String contentLengthString = headers.getValue("Content-Length");
        if (StringUtils.isEmpty(contentLengthString)) {
            return contentLength;
        }

        try {
            contentLength = Long.parseLong(contentLengthString);
        } catch (NumberFormatException | NullPointerException e) {
            logger.warning("Could not parse the HTTP header content-length: '{}'.",
                    headers.getValue("content-length"), e);
        }

        return contentLength;
    }

    private boolean shouldBodyBeLogged(String contentTypeHeader, long contentLength) {
        return !FormatType.RAW.equalsIgnoreCase(contentTypeHeader)
                && contentLength != 0
                && contentLength < MAX_BODY_LOG_SIZE;
    }

    private static String convertStreamToString(ByteArrayOutputStream stream, ClientLogger logger) {
        try {
            return stream.toString("UTF-8");
        } catch (UnsupportedEncodingException ex) {
            throw logger.logExceptionAsError(new RuntimeException(ex));
        }
    }

    private static CompletableFuture<ByteBuffer> writeBufferToBodyStream(WritableByteChannel channel, ByteBuffer byteBuffer) throws IOException {
        channel.write(byteBuffer.duplicate());
        return CompletableFuture.completedFuture(byteBuffer);
    }
}
